<?php

class LetsSyncroLLC_Oct8ne_Helper_Data extends Mage_Core_Helper_Abstract {

    static public function getProductsSummaryByIds($productIds) {
        if (empty($productIds)) {
            return array();
        }

        $attributes = Mage::getSingleton('catalog/config')->getProductAttributes();
        $products = Mage::getModel('catalog/product')
                ->getCollection()
                ->addAttributeToFilter('entity_id', array('in' => $productIds))
                ->addAttributeToSelect($attributes)
                ->addAttributeToSelect('image')
                ->addAttributeToSelect('small_image')
                ->addAttributeToSelect('thumbnail')
                ->addFinalPrice();

        // Store the products in a dictionary
        $productDict = array();
        foreach ($products as $product) {
            $id = $product->getId();
            $productDict[$id] = self::createProductSummaryFromProduct($product);
        }

        // Return the products in the same order that they were queried
        $result = array();
        foreach ($productIds as $id) {
            if (array_key_exists($id, $productDict)) {
                $result[] = $productDict[$id];
            }
        }
        return $result;
    }

    static public function getProductsInfoByIds($productIds, $qtyPerProduct = null) {
        if (empty($productIds)) {
            return array();
        }
        $profiler = self::getProfiler(); /* @var $profiler LetsSyncroLLC_Oct8ne_Helper_Debug */
        $startTime = $profiler->startProfile();

        if (Mage::getStoreConfig('oct8neData/advancedDataConfig/loadCompleteProductInfo')) {
            $result = self::getFullProductsInfoByIds($productIds, $qtyPerProduct);
        } else {
            $result = self::getOptimizedProductsInfoByIds($productIds, $qtyPerProduct);
        }
        $profiler->endProfile($startTime, "[Data] Create product info");
        return $result;
    }

    static private function getFullProductsInfoByIds($productIds, $qtyPerProduct = null) {
        $result = array();
        foreach ($productIds as $id) {
            $product = Mage::getModel('catalog/product')->load($id);
            $qty = $qtyPerProduct && array_key_exists($id, $qtyPerProduct) ? $qtyPerProduct[$id] : null;
            $result[] = self::createProductInfoFromProduct($product, $qty);
        }
        return $result;
    }

    static private function getOptimizedProductsInfoByIds($productIds, $qtyPerProduct = null) {
        $attributes = Mage::getSingleton('catalog/config')->getProductAttributes();
        $products = Mage::getModel('catalog/product')
                ->getCollection()
                ->addAttributeToFilter('entity_id', array('in' => $productIds))
                ->addAttributeToSelect($attributes)
                ->addAttributeToSelect('image')
                ->addAttributeToSelect('small_image')
                ->addAttributeToSelect('thumbnail')
                ->addFinalPrice();

        // Add product description attributes
        self::addProductDescriptionsToQuery($products);

        // Add stock information
        Mage::getModel('catalogInventory/stock')->addItemsToProducts($products);

        // Preloads the media gallery for every product 
        if (Mage::getStoreConfig('oct8neData/advancedDataConfig/imageGalleryPrefetch')) {
            self::addMediaGalleryAttributeToCollection($products);
        }

        $productDict = array();
        foreach ($products as $product) {
            $id = $product->getId();
            $qty = $qtyPerProduct && array_key_exists($id, $qtyPerProduct) ? $qtyPerProduct[$id] : null;
            $productDict[$id] = self::createProductInfoFromProduct($product, $qty);
        }

        // Return the products in the same order that they were queried
        $result = array();
        foreach ($productIds as $id) {
            if (array_key_exists($id, $productDict)) {
                $result[] = $productDict[$id];
            }
        }
        return $result;
    }

    static public function getProductInfo($product_id, $qty = null) {
        $product = Mage::getModel('catalog/product')->load($product_id);
        return self::createProductInfoFromProduct($product, $qty);
    }

    static public function createProductSummaryFromProduct($product, $qty = null) {
        $product_id = $product->getId();
        //$formattedPrevPrice = $product->getFinalPrice() != 0 ? Mage::helper('core')->currency($product->getPrice(), true, false): NULL;  		      
        //$formattedPrice = $product->getFinalPrice() != 0 ? Mage::helper('core')->currency($product->getFinalPrice(), true, false) : NULL;
        $formattedPrevPrice = $product->getFinalPrice() != 0 ? 
            Mage::helper('core')->currency(
                Mage::helper('tax')->getPrice($product, $product->getPrice()),
                true,
                false
            ) : null;
        $formattedPrice = $product->getFinalPrice() != 0 ?
            Mage::helper('core')->currency(
                Mage::helper('tax')->getPrice($product, $product->getFinalPrice()),
                true,
                false
            ) : null;    
            
            
            
        $productUrl = self::_remove_http($product->getProductUrl());
        $thumbnail = self::_remove_http(self::getProductThumbnail($product, 120));
        $title = htmlspecialchars($product->getName(), ENT_QUOTES, "UTF-8");
        

        $result = array(
            'internalId' => $product_id,
            'title' => $title,
            'formattedPrevPrice' => $formattedPrevPrice,
            'formattedPrice' => $formattedPrice,
            'productUrl' => $productUrl,
            'thumbnail' => $thumbnail,
        );
        if ($qty) {
            $result['qty'] = $qty;
        }
        return $result;
    }
    
    public static function getCart() {
        $quote = Mage::getSingleton('checkout/session')->getQuote();
        $items = $quote->getAllVisibleItems();
        $cart = array();
        $totalItems = 0;
        foreach ($items as $item) {
            $product = $item->getProduct();
            $totalItems += $item->getQty();
            $cart[] = array(
                "internalId" => $product->getId(),  
                "title" => htmlspecialchars($product->getName(), ENT_QUOTES, "UTF-8"),
                "qty" => $item->getQty(),
                "price" => $product->getPrice()
            );
        }
        $result = array(
            "price" => $quote->getSubtotal(),
            "finalPrice" => $quote->getGrandTotal(),
            "currency" => $quote->getQuoteCurrencyCode(),
            "totalItems" => $totalItems,
            "cart" => $cart,
        );
        return $result;
    }

    static public function createProductInfoFromProduct($product, $qty = null) {

        $profiler = self::getProfiler(); /* @var $profiler LetsSyncroLLC_Oct8ne_Helper_Debug */

        $tmp = self::createProductSummaryFromProduct($product);

        $description = self::getSanitizedDescription($product);

        // useProductUrl must be 
        // false if the product has stock and the product has no attributes (not configurable)
        // true if the product is out of stock or the product has attributes (configurable)        
        $useProductUrl = true;
        if (self::isStockAvailable($product)) {
            if (!self::productHasOptions($product)) {
                $useProductUrl = false;
            }
        }

        $start = $profiler->startProfile('[Data]  - Start getting medias');
        $medias = self::getProductImagesUrl($product);
        $profiler->endProfile($start, '[Data]  - End get medias');

        // If medias is empty, return the thumbnail as unique main image
        if ((count($medias) == 0) && array_key_exists("thumbnail", $tmp)) {
            $medias = array(
                'url' => $tmp["thumbnail"]
            );
        }

        $start = $profiler->startProfile();
        if (!is_null($qty)) {
            $tmp['qty'] = $qty;
        }
        $result = array_merge($tmp, array(
            'description' => trim($description),
            'rating' => 'no-rating',
            'addToCartUrl' => self::getProductBuyUrl($product),
            'useProductUrl' => $useProductUrl,
            'medias' => $medias,
        ));
        $profiler->endProfile($start, '[Data]  - Compose ProductInfo structure');

        return $result;
    }

    static private function isStockAvailable($product) {

        $stockItem = $product->getStockItem();
        if (!$stockItem) {
            $stockItem = Mage::getModel('cataloginventory/stock_item')->loadByProduct($product);
        }

        $available = true;
        $productIsInStock = $stockItem->getIsInStock();
        $productsCount = (float) $stockItem->getQty();
        $minProductsAllowed = (float) $stockItem->getMinQty();
        $backOrdersAllowed = $stockItem->getBackorders();
        $magentoIsManagingStocks = $stockItem->getManageStock();
        $ignoreBackorders = Mage::getStoreConfig('oct8neData/productData/ignoreBackorders');

        if ($magentoIsManagingStocks) {                                         
            if (!$productIsInStock || $productsCount <= $minProductsAllowed) {  
                if ($ignoreBackorders) {
                    $available = false;                                         
                } else {
                    $available = $stockItem->getBackorders() ? true : false;    
                }
            }
        }                                                                       
        return $available;
    }

    static public function getRelatedProducts($productId, $page, $pageSize) {

        $product = Mage::getModel('catalog/product')->load($productId);

        // First, try to get related products specified by admins
        $relatedCollection = $product->getRelatedProductCollection()
                ->AddStoreFilter()
                ->addAttributeToSelect('id')
                ->addAttributeToSort('price', 'asc')
                ->setPageSize($pageSize)
                ->setCurPage($page);

        if ($relatedCollection->count() > 0) {
            $relatedProductIds = array();
            foreach ($relatedCollection as $relatedProduct) {
                $relatedProductIds[] = $relatedProduct->getId();
            }
            return self::loadRelatedProductsFromCollectionOfIds($relatedProductIds, $page, $pageSize);
        }

        // If no related products specified by admins, ask the search engine for them
        $searchEngineName = strtolower(Mage::getStoreConfig('oct8neSearch/searchEngine/engine'));
        if ($searchEngineName == '') {
            $searchEngineName = 'Magento';
        }
        $searchEngine = Mage::Helper('oct8ne/search_' . $searchEngineName); /* @var $searchEngine LetsSyncroLLC_Oct8ne_Helper_Search_Base */
        $productIds = $searchEngine->getRelatedProductIds($product, $page, $pageSize);
        return self::loadRelatedProductsFromCollectionOfIds($productIds, $page, $pageSize);
    }

    static private function loadRelatedProductsFromCollectionOfIds($relatedProductIds, $page, $pageSize) {
        // Until the coviewer supports pagination, just
        // return as total the number of related products in the first page
        $productsCount = sizeof($relatedProductIds);
        $result = array(
            'total' => $productsCount,
            'results' => self::getProductsSummaryByIds($relatedProductIds)
        );

        return $result;
    }

    static public function getProductThumbnail($product, $width) {
        $imageHelper = self::getImageHelper();
        return $imageHelper->getProductThumbnail($product, $width);
    }

    static private function getSanitizedDescription($product) {
        $descriptionAttrs = Mage::getStoreConfig('oct8neData/productData/productDataDescription');
        if (trim($descriptionAttrs) == '') {
            $groups = self::getDefaultDescriptionAttributes();
        } else {
            $groups = explode(',', $descriptionAttrs); // Example: ["a+b", "c"]
        }
        $description = "";
        foreach($groups as $group) {
            $fields = explode('+', $group);
            foreach($fields as $field) { 
                $value = $product[$field];
                if(!is_null($value) && strlen($value)>0) {
                    if (strlen($description) > 0) {
                        $description .= "<hr class='oct8ne-description-separator' />";
                    }
                    $description .= $value;
                }
            }
            if(strlen($description) > 0)
                break;
        }
        if (!is_null($description)) {
            $description = htmlspecialchars($description, ENT_QUOTES, "UTF-8");
            $description = trim($description, "\r\n\t ");
            $description = str_replace("\r", "", $description);
            $description = str_replace("\t", "", $description);
            $description = str_replace("\n", "&lt;br/ &gt;", $description);
        }
        return $description;
    }

    static private function _remove_http($str) {
        return preg_replace('#^https?:#', '', $str);
    }

    static private function getProductBuyUrl($product) {
        $full_url = Mage::helper('checkout/cart')->getAddUrl($product);
        $buy_url = preg_replace('#^https?:#', '', $full_url);
        return $buy_url;
    }

    // TODO: Simplify this method
    static private function productHasOptions($product) {
        $has_options = false;

        /** Options * */
        $product_options = $product->getTypeInstance(true)->hasOptions($product);
        $has_options = ($product_options ? true : $has_options);

        /** Grouped Product * */
        $product_grouped = $product->getTypeId();
        $has_options = ($product_grouped == "grouped" ? true : $has_options);

        return $has_options;
    }

    static private function getProductImagesUrl($product) {
        $helper = self::getImageHelper(); /* @var $helper LetsSyncroLLC_Oct8ne_Helper_Image_Default */
        return $helper->getProductImagesUrl($product);
    }

    // Loads the media for all products in just one query
    // Hack found in http://stackoverflow.com/questions/7890616/get-product-media-gallery-images-from-a-product-collection-in-magento
    private static function addMediaGalleryAttributeToCollection(Mage_Catalog_Model_Resource_Product_Collection $_productCollection) {
        $_mediaGalleryAttributeId = Mage::getSingleton('eav/config')->getAttribute('catalog_product', 'media_gallery')->getAttributeId();
        $_read = Mage::getSingleton('core/resource')->getConnection('catalog_read');

        $pageCur = $_productCollection->getCurPage();
        $pageSize = $_productCollection->getPageSize();
        $offset = $pageSize * ($pageCur - 1);

        $ids = $_productCollection->getAllIds($pageSize, $offset);

        $galleryTable = Mage::getConfig()->getTablePrefix() . 'catalog_product_entity_media_gallery';
        $galleryValuesTable = Mage::getConfig()->getTablePrefix() . 'catalog_product_entity_media_gallery_value';
        if (count($ids) > 0) {
            $sql = '
                    SELECT
                        main.entity_id, `main`.`value_id`, `main`.`value` AS `file`, `value`.`disabled`,
                        `value`.`position` AS `position`, `default_value`.`position` AS `default_position`, 
                        `default_value`.`disabled` AS `disabled_default`
                    FROM `' . $galleryTable . '` AS `main`
                        LEFT JOIN `' . $galleryValuesTable . '` AS `value`
                            ON main.value_id=value.value_id AND value.store_id=' . Mage::app()->getStore()->getId() . '
                        LEFT JOIN `' . $galleryValuesTable . '` AS `default_value`
                            ON main.value_id=default_value.value_id AND default_value.store_id=0
                    WHERE (
                        main.attribute_id = ' . $_read->quote($_mediaGalleryAttributeId) . ')
                        AND (main.entity_id IN (' . $_read->quote($_productCollection->getAllIds()) . '))
                    ORDER BY value.position, default_position
                ';
            $_mediaGalleryData = $_read->fetchAll($sql);

            $_mediaGalleryByProductId = array();
            foreach ($_mediaGalleryData as $_galleryImage) {
                $k = $_galleryImage['entity_id'];
                unset($_galleryImage['entity_id']);
                if (!isset($_mediaGalleryByProductId[$k])) {
                    $_mediaGalleryByProductId[$k] = array();
                }
                $_mediaGalleryByProductId[$k][] = $_galleryImage;
            }
            unset($_mediaGalleryData);
            foreach ($_productCollection as &$_product) {
                $_productId = $_product->getData('entity_id');
                if (isset($_mediaGalleryByProductId[$_productId])) {
                    $_product->setData('media_gallery', array('images' => $_mediaGalleryByProductId[$_productId]));
                }
            }
            unset($_mediaGalleryByProductId);
        }
        return $_productCollection;
    }

    private static function addProductDescriptionsToQuery($products) {
        $attrs = self::getDescriptionAttributes();
        foreach ($attrs as $attr) {
            if (trim($attr) != '') {
                $products->addAttributeToSelect($attr);
            }
        }
    }
    
    private static function getDefaultDescriptionAttributes() {
        return array('description', 'short_description');
    }
    
    private static function getDescriptionAttributes() {
        $descriptionAttrs = Mage::getStoreConfig('oct8neData/productData/productDataDescription');
        if (trim($descriptionAttrs) == '') {
            $result = self::getDefaultDescriptionAttributes();
        } else {
            $attrs = array();
            $groups = explode(',', $descriptionAttrs); // Example: ["a+b", "c"]
            foreach($groups as $group) { 
                $fields = explode('+', $group);
                foreach($fields as $field) {
                    $field = trim($field);
                    if(!array_key_exists($field, $attrs) ){
                        $attrs[$field] = null;
                    }
                }
            }
            $result = array_keys($attrs);
        }
        return $result;
    }

    private static $profiler = NULL;

    private static function getProfiler() {
        if (is_null(self::$profiler)) {
            self::$profiler = Mage::helper('oct8ne/debug');
        }
        return self::$profiler;
    }

    static private function getImageHelper() {
        $helperName = Mage::getStoreConfig('oct8neData/advancedDataConfig/imageHelperName');
        if (!$helperName) {
            $helperName = 'default';
        }
        $helperName = 'oct8ne/image_' . $helperName;
        return Mage::helper($helperName);
    }

}
