<?php

/* Celebros Search engine adapter
 * Parameters 
 * - host: Celebros host (not including protocol)
 * - port: port
 * - siteKey: Site key
 * - id_field: Field containing the product id
 * - name_field: Field containing the product name. Used for sorting.
 * - price_field: Field containing the price. Used for sorting.
 * 
 * Example:
 * host=http://xxx.xxx.xxx;port=6035;siteKey=abcde;id_field=mag_id;name_field=Title
 * 
 */
/*
  host=ezupv8-search.celebros.com;
  port=6035;
  siteKey=EZupV8;
  id_field=mag_id;
  name_field=Title;
  price_field=Price;
 */

include_once("CelebrosAPI/QwiserSearchAPI.php");

class LetsSyncroLLC_Oct8ne_Helper_Search_Celebros extends LetsSyncroLLC_Oct8ne_Helper_Search_Base {

    public function getEngineName() {
        return "celebros";
    }

    public function isValidSearchCriteria($searchTerm) {
        if (is_null($searchTerm) || strlen($searchTerm) == 0) {
            return false;
        }
        return true;
    }

    public function search($storeId, $searchTerm, $searchOrder, $searchDir, $page, $pageSize, &$totalSearchResults, &$attrs_applied, &$attrs_available) {
        $handlerCookieName = "shandler";
        $this->log("Searching " . $searchTerm);

        $appliedFiltersDictionary = $this->getAppliedFilters();

        if (($page > 1) && isset($_COOKIE[$handlerCookieName])) { // Change the page of the last query
            $searchHandle = $_COOKIE[$handlerCookieName];
            $result = $this->moveToPage($searchHandle, $page - 1);
        } else if (($appliedFiltersDictionary) && isset($_COOKIE[$handlerCookieName])) { // We must apply filters on the last
            $searchHandle = $_COOKIE[$handlerCookieName];
            $result = $this->applyFilters($searchHandle, $appliedFiltersDictionary);
        } else {
            $result = $this->executeFullCelebrosSearchQuery($searchTerm, NULL, $searchOrder, $searchDir, $page, $pageSize, TRUE /* addFacets */);
        }

        // Return if no results found
        if (!$result) {
            return array();
        }

        // If we are browing the first page, we must store the search handle 
        // in a session cookie for reusing it in the next requests.
        if ($page == 1 && !$appliedFiltersDictionary) {
            $searchHandle = $result->QwiserSearchResults->myDOMNode->getAttribute("SearchHandle");
            setcookie($handlerCookieName, $searchHandle, 0);
        }

        $totalSearchResults = $result->QwiserSearchResults->myDOMNode->getAttribute("RelevantProductsCount");
        $currentPage = $result->SearchInformation->CurrentPage;
        $productIds = $this->getProductIds($result);
        $allAvailableFilters = $this->getAvailableFilters($result);
        $attrs_applied = $this->getResponseAppliedFilters($allAvailableFilters, $appliedFiltersDictionary);
        $attrs_available = $this->getAvailableButNotAppliedFilters($attrs_applied, $allAvailableFilters);

        $this->log("Total {$totalSearchResults}, ids of this page: [" . implode(",", $productIds) . "]");
        return $productIds;
    }

    public function getRelatedProductIds($product, $page, $pageSize) {
        
        // Improvement: we can get the answers for this product (function GetProductAnswers()), 
        // and then try performing searches using this answers as criteria until we get results
        
        $name = $product->getName();
        $term = "";
        foreach (explode(" ", $name) as $word) {
            $trimmed = trim($word);
            if ($trimmed != "" && strlen($trimmed) >= 2) {
                if ($term != "") {
                    $term .= " OR ";
                }
                $term .= $word;
            }
        }
        $result = $this->executeFullCelebrosSearchQuery($term, NULL, 'score', 'asc', $page, $pageSize, FALSE /* addFacets */);
        if(!$result) {
            return array();
        }
        $productIds = $this->getProductIds($result);
        return $productIds;
    }

    private function moveToPage($searchHandle, $page) {
        $api = $this->createApi();
        $result = $api->moveToPage($searchHandle, $page);
        if (!$result) {
            $this->log("Move to page without results. Last operation message: $api->LastOperationErrorMessage");
        }
        return $result;
    }

    private function applyFilters($searchHandle, $appliedFilters) {
        $api = $this->createApi();

        $filters = "";
        $separator = urlencode("#");
        foreach ($appliedFilters as $param => $value) {
            if ($filters != "") {
                $filters .= $separator;
            }
            $filters .= $value;
        }
        $result = $api->AnswerQuestions($searchHandle, $filters, "1");
        if (!$result) {
            $this->log("Apply filters without results. Last operation message: $api->LastOperationErrorMessage");
        }
        return $result;
    }

    private function executeFullCelebrosSearchQuery($searchTerm, $searchCategories, $searchOrder, $searchDir, $page, $pageSize, $addFacets) {

        $api = $this->createApi();
        $searchProfile = "";
        $answerId = "";
        $effectOnSearchPath = "1"; // * 0-2, PDF pag 17 
        $priceColumn = ""; // Optional?
        $sortingField = $this->getCelebrosSort($searchOrder);
        $bNumericSort = $searchOrder=="price"? "1": "0";
        $bAscending = $searchDir == "asc"? "1": "0";

        if ($addFacets) {
            $applied = $this->getAppliedFilters();
            foreach ($applied as $param => $value) {
                $answerId = $value;
                break;
            }
        }

        $result = $api->SearchAdvance($searchTerm, $searchProfile, $answerId, $effectOnSearchPath, $priceColumn, $pageSize, $sortingField, $bNumericSort, $bAscending);
        if (!$result) {
            $this->log("No results. Last operation message: $api->LastOperationErrorMessage");
        }
        return $result;
    }

    private function createApi() {
        $port = $this->getEngineParam("port");
        $host = $this->getEngineParam("host");
        $siteKey = $this->getEngineParam("siteKey");
        $api = new QwiserSearchAPI($siteKey, $host, $port);
        return $api;
    }

    private function getProductIds($result) {
        $productIds = array();
        $idField = $this->getEngineParam('id_field');

        if (!$result || !$result->Products)
            return $productIds;

        $products = $result->Products->Items;
        foreach ($products as $product) {
            $productIds[] = $product->Field[$idField];
        }

        // Uncomment for local testing
        // $productIds = array('906', '875', '874', '554', '553', '552', '551', '549', '399', '398');

        return $productIds;
    }

    private function getCelebrosSort($searchOrder) {
        $result = "";
        if (!$searchOrder) {
            $searchOrder = "relevance";
        }
        switch ($searchOrder) {
            case "score":
            case "relevance":
                $result = ""; // By default, the order is "relevance"
                break;
            case "price":
                $result = $this->getEngineParam("price_field");
                break;
            case "name":
                $result = $this->getEngineParam("name_field");
        }
        return $result;
    }

    private function getAvailableFilters($result) {
        $filters = array();
        $questions = $result->Questions;
        foreach ($questions->Items as $question) {
            $questionName = $question->Text;
            $questionId = "a-" . $question->Id;
            $answers = $question->Answers;
            $options = array();
            foreach ($answers->Items as $answer) {
                $answerId = $answer->Id;
                $answerName = $answer->Text;
                $answerCount = $answer->ProductCount;
                $options[] = $this->createFilterOption($answerName, $answerId, $answerCount);
            }
            $filters[] = $this->createFilterInfo($questionId, $questionName, $options);
        }
        return $filters;
    }

    private function getResponseAppliedFilters($availableFilters) {
        $filters = array();
        $request = $this->getRequest();

        foreach ($availableFilters as $availableFilter) {
            $parameterName = $availableFilter["param"];
            $parameterLabel = $availableFilter["paramLabel"];
            $parameterValue = $request->getParam($parameterName);
            if (!is_null($parameterValue)) {
                foreach ($availableFilter["options"] as $option) {
                    if ($option["value"] == $parameterValue) {
                        $filters[] = $this->createAppliedFilter(
                                $parameterName, $parameterLabel, $parameterValue, $option["valueLabel"]
                        );
                        break;
                    }
                }
            }
        }
        return $filters;
    }
}
