<?php

/* SearchString Search engine adapter
 * Parameters 
 * - siteId: site Id (The siteId parameter identifies your SearchSpring account.  You can find your siteId in the SearchSpring Management Console on the My Account page (click here to go to the SearchSpring Management Console).)
 * - protocol: http || https 
 */

class LetsSyncroLLC_Oct8ne_Helper_Search_Searchspring extends LetsSyncroLLC_Oct8ne_Helper_Search_Base {
    public function getEngineName() {
        return "searchspring";
    }
    
    const FilterRangeSeparator = ":::";

    public function isValidSearchCriteria($searchTerm) {
        if (is_null($searchTerm) || strlen($searchTerm) == 0) {
            return false;
        }
        return true;
    }

    public function search($storeId, $searchTerm, $searchOrder, $searchDir, $page, $pageSize, &$totalSearchResults, &$attrs_applied, &$attrs_available) {
        $this->log("Searching " . $searchTerm);
        $result = $this->executeSearchSpringSearchQuery($searchTerm, NULL, $searchOrder, $searchDir, $page, $pageSize);
        if (is_null($result)) {
            return array();
        }
        $totalSearchResults = $result["pagination"]["totalResults"];
        $productIds = $this->getProductIds($result);

        $allAvailableFilters = $this->getAvailableFilters($result);
        $attrs_applied = $this->getResponseAppliedFilter($allAvailableFilters);
        $attrs_available = $this->getAvailableButNotAppliedFilters($attrs_applied, $allAvailableFilters);
        
//        if (!empty($productIds)) {
//            // Uncomment for local testing
//            $productIds = array('906', '875', '874', '554', '553', '552', '551', '549', '399', '398');
//        }
        return $productIds;
    }

    public function getRelatedProductIds($product, $page, $pageSize) {
        $name = $product->getName();        
        if (!empty($name)) {
            $result = $this->executeSearchSpringSearchQuery($name, NULL, 'relevance', 'asc', $page, $pageSize, FALSE);
        }        
        $productIds = $this->getProductIds($result);        
//        if (!empty($productIds)) {
//            // Uncomment for local testing
//            $productIds = array('906', '875', '874', '554', '553', '552', '551', '549', '399', '398');
//        }
        return $productIds;
    }

    private function executeSearchSpringSearchQuery($searchTerm, $searchCategories, $searchOrder, $searchDir, $page, $pageSize, $includeFacets = TRUE, $method = NULL) {
        if (!function_exists('curl_init')) {
            return NULL;
        }
         $host = $this->getSearchSpringProtocol()."://api.searchspring.net/api/search/search.json";
        $queryParams = array(
            'siteId' => $this->getEngineParam("siteId"),
            'resultsFormat' => "native",//response results will be formatted as native JSON objects
            'q' => $searchTerm,
            'page' => $page,
            'resultsPerPage' => $pageSize,   
            'sort.'.$this->getSearchSpringSort($searchOrder) => $searchDir,
        );
        if (!is_null($method)) {
            $queryParams["method"] = $method;
        }
                
        if ($includeFacets) {
            $queryParams = $this->getFacets($queryParams);
        }

        $query = $host . '?' . http_build_query($queryParams);
        $this->log("Executing query: {$query}");
        $request = curl_init($query);
        curl_setopt($request, CURLOPT_POST, 1);
        curl_setopt($request, CURLOPT_RETURNTRANSFER, 1);
        $response = curl_exec($request);
        $err = curl_error($request);
        if (is_null($err) || $err == "") {
            $result = json_decode($response, true);
            if (is_null($result)) {
                $contentType = curl_getinfo($request, CURLINFO_CONTENT_TYPE);
                $msg = "SearchSpring search is not returning valid JSON data, or result is empty. Content-type returned: " . $contentType;
                Mage::log("[Oct8ne] " . $msg);
                $this->log($msg);
            } else {
                $this->log("Query result received");
            }
        } else {
            $this->log("Search error: " . $err);
            $result = NULL;
        }
        curl_close($request);
        return $result;
    }

    private function getProductIds($result) {
        $productIds = array();
        foreach ($result["results"] as $product) {
            $productIds[] = $product["uid"];
        }
        return $productIds;
    }
    
    private function getAvailableFilters($result) {
        $filters = array();
        foreach ($result["facets"] as $facet) {
            if (key_exists("parent_id", $facet)) {
                continue; // Ignore sub-facets
            }

            $options = array();
            foreach ($facet["values"] as $value) {
                if(isset($value["low"]) && isset($value["high"])){                    
                    $options[] = $this->createFilterOption($value["label"], urlencode( $value["low"].self::FilterRangeSeparator.$value["high"]) , $value["count"]);
                }else{
                    $options[] = $this->createFilterOption($value["label"], $value["value"], $value["count"]);
                }
            }
            $filters[] = $this->createFilterInfo($facet["field"], $facet["label"], $options);
        }
        return $filters;
    }
    
    private function getResponseAppliedFilter($availableFilters) {
        $filters = array();
        $request = $this->getRequest();
        foreach ($availableFilters as $availableFilter) {
            $filterParameterValue = $request->getParam($availableFilter["param"]);
            if (!is_null($filterParameterValue)) {
                $options = $availableFilter["options"];
                $currentLabel = $this->getCurrentOptionLabel($filterParameterValue, $options);
                $filters[] = $this->createFilterInfo(
                        $availableFilter["param"], urldecode($availableFilter["paramLabel"]), $options, $currentLabel, $filterParameterValue
                );
            }
        }
        return $filters;
    }
    
    private function getSearchSpringSort($searchOrder) {
        $result = "";
        if (!$searchOrder) {
            $searchOrder = "relevance";
        }
        switch ($searchOrder) {
            case "score":
                $result = "relevance";
                break;
            case "relevance":
                $result = "relevance"; // By default, the order is "relevance"
                break;
            case "price":
                $result = "price";
                break;
            case "name":
                $result = "name";
        }
        return $result;
    }
    
    function getSearchSpringProtocol() {
        $protocol = $this->getEngineParam("protocol");
        if(is_null($protocol)){
            $protocol = "https";            
        }
        return $protocol;
    }
    
    private function getFacets($queryParams) {

        $appliedFilters = $this->getAppliedFilters();
        if ($appliedFilters) {
            $facet = "";
            foreach ($appliedFilters as $param => $value) {
                if (!is_null($value) && trim($value) != '') {
                    $value = urldecode($value);
                    if(strpos($value,self::FilterRangeSeparator) === false){
                        $queryParams["filter.".$param] = $value;
                    }else{
                        $params = explode(self::FilterRangeSeparator, $value);
                        if(count($params)>1){
                            $queryParams["filter.".$param.".low"] = $params[0];
                            $queryParams["filter.".$param.".high"] = $params[1];
                        }else if(count($params)>0){
                            $queryParams["filter.".$param.".low"] = $params[0];
                        }     
                    }
                }
            }
        }
        return $queryParams;
    }
    
    private function getCurrentOptionLabel($value, $options) {
        foreach ($options as $option) {            
            if ($option['value'] == $value) {
                return $option["valueLabel"];
            }
        }
        return $value;
    }
}
