<?php

/* SLI Search engine adapter
 * 
 * Parameter: host (SLI endpoint)
 * 
 */

class LetsSyncroLLC_Oct8ne_Helper_Search_Sli extends LetsSyncroLLC_Oct8ne_Helper_Search_Base {

    public function getEngineName() {
        return "sli";
    }

    public function isValidSearchCriteria($searchTerm) {
        if (is_null($searchTerm) || strlen($searchTerm) == 0) {
            return false;
        }
        return true;
    }

    public function search($storeId, $searchTerm, $searchOrder, $searchDir, $page, $pageSize, &$totalSearchResults, &$attrs_applied, &$attrs_available) {
        $this->log("Searching " . $searchTerm);
        $result = $this->executeSliSearchQuery($searchTerm, NULL, $searchOrder, $searchDir, $page, $pageSize);
        if (is_null($result)) {
            return array();
        }
        $totalSearchResults = $result["result_meta"]["total"];
        
        $allAvailableFilters = $this->getAvailableFilters($result);
        $attrs_applied = $this->getResponseAppliedFilter($allAvailableFilters);
        $attrs_available = $this->getAvailableButNotAppliedFilters($attrs_applied, $allAvailableFilters);

        $productIds = $this->getProductIds($result);
        $this->log("Total {$totalSearchResults}, ids of this page: [" . implode(",", $productIds) . "]");
        return $productIds;
    }

    public function getRelatedProductIds($product, $page, $pageSize) {
        $categoryIds = $product->getCategoryIds();
        while (count($categoryIds) > 2) {
            array_pop($categoryIds);
        }
        if (!empty($categoryIds)) {
            $result = $this->executeSliSearchQuery(NULL, $categoryIds, 'score', 'asc', $page, $pageSize, FALSE);
        } else {
            $name = $product->getName();
            $result = $this->executeSliSearchQuery($name, NULL, 'score', 'asc', $page, $pageSize, FALSE, 'or');
        }
        $productIds = $this->getProductIds($result);
        return $productIds;
    }

    private function executeSliSearchQuery($searchTerm, $searchCategories, $searchOrder, $searchDir, $page, $pageSize, $includeFacets = TRUE, $method = NULL) {
        if (!function_exists('curl_init')) {
            return NULL;
        }
        $host = $this->normalizeHost($this->getEngineParam("host"));
        $queryParams = array(
            'ts' => 'json-full',
            'w' => $searchTerm,
            'cnt' => $pageSize,
            'srt' => $pageSize * ($page - 1),
            'isort' => $this->getSliSort($searchOrder, $searchDir),
            'usecache' => 'no'
        );
        if ($searchCategories) {
            $queryParams['catfilter'] = implode('^', $searchCategories);
            $queryParams["w"] = '*';
        }

        if ($includeFacets) {
            $queryParams['af'] = $this->getFacets();
        }
        if (!is_null($method)) {
            $queryParams["method"] = $method;
        }

        $query = $host . '/search?' . http_build_query($queryParams);

        $this->log("Executing query: {$query}");

        $request = curl_init($query);
        curl_setopt($request, CURLOPT_POST, 1);
        curl_setopt($request, CURLOPT_RETURNTRANSFER, 1);
        $response = curl_exec($request);
        $err = curl_error($request);
        if (is_null($err) || $err == "") {
            $result = json_decode($response, true);
            if (is_null($result)) {
                $contentType = curl_getinfo($request, CURLINFO_CONTENT_TYPE);
                $msg = "SLI search is not returning valid JSON data, or result is empty. Content-type returned: " . $contentType;
                Mage::log("[Oct8ne] " . $msg);
                $this->log($msg);
            } else {
                $this->log("Query result received");
            }
        } else {
            $this->log("Search error: " . $err);
            $result = NULL;
        }
        curl_close($request);
        return $result;
    }

    private function getProductIds($result) {
        $productIds = array();
        foreach ($result["results"] as $product) {
            $productIds[] = $product["product_id"];
        }

        // TODO: Remove
        // $productIds = array('142', '1403', '1639', '1919', '3034', '5629', '7740', '10121', '17952', '23361');       

        return $productIds;
    }

    function normalizeHost($host) {
        $host = rtrim($host, '/');
        if (!$this->startsWith($host, 'http://') && !!$this->startsWith($host, 'https://')) {
            $host = 'http://' . $host;
        }
        return $host;
    }

    function endsWith($str, $subStr) {
        $length = strlen($subStr);
        if ($length == 0) {
            return true;
        }
        return (substr($str, -$length) === $subStr);
    }

    function startsWith($str, $subStr) {
        $length = strlen($subStr);
        return (substr($str, 0, $length) === $subStr);
    }

    private function getSliSort($searchOrder, $searchDir) {
        $result = "";
        switch ($searchOrder) {
            case "score":
                $result .= "score";
                break;
            case "relevance":
                $result .= "rel";
                break;
            case "price":
                $result .= "price";
                break;
            case "name":
                $result .= "title";
        }
        if ($searchDir == "desc") {
            $result .= " rev";
        }
        return $result;
    }

    private function getAvailableFilters($result) {
        $filters = array();
        foreach ($result["facets"] as $facet) {
            if (key_exists("parent_id", $facet)) {
                continue; // Ignore sub-facets
            }

            $options = array();
            foreach ($facet["values"] as $value) {
                $options[] = $this->createFilterOption($value["name"], $value["id"], $value["count"]);
            }
            $filters[] = $this->createFilterInfo($facet["id"], $facet["name"], $options);
        }
        return $filters;
    }

    private function getResponseAppliedFilter($availableFilters) {
        $filters = array();
        $request = $this->getRequest();
        foreach ($availableFilters as $availableFilter) {
            $filterParameterValue = $request->getParam($availableFilter["param"]);
            if (!is_null($filterParameterValue)) {
                $options = $availableFilter["options"];
                $currentLabel = $this->getCurrentOptionLabel($filterParameterValue, $options);
                $filters[] = $this->createFilterInfo(
                        $availableFilter["param"], $availableFilter["paramLabel"], $options, $currentLabel, $filterParameterValue
                );
            }
        }
        return $filters;
    }

    private function getCurrentOptionLabel($value, $options) {
        foreach ($options as $option) {
            if ($option['value'] == $value) {
                return $option["valueLabel"];
            }
        }
        return $value;
    }

    private function getFacets() {

        $appliedFilters = $this->getAppliedFilters();
        if ($appliedFilters) {
            $facet = "";
            foreach ($appliedFilters as $param => $value) {
                if (!is_null($value) && trim($value) != '') {
                    $facet .= $param . ':' . $value . ' ';
                }
            }
        }
        return trim($facet);
    }

    protected function log($message) {
        $debug = Mage::helper('oct8ne/debug'); /* @var $debug LetsSyncroLLC_Oct8ne_Helper_Debug */
        $engineName = $this->getEngineName();
        $debug->log("[Search-{$engineName}] " . $message);
    }

}
