<?php

/* Solr Search engine adapter
 * Parameters 
 * - host: Solr endpoint
 * - username: username (optional)
 * - password: password (optional)
 * - id_field: name of id field
 * - name_field: name of product name field (used for sorting)
 * - price_field: name of price field (used for sorting and filtering)
 * - relevance_field: name of relevance field
 * - query_fields: name of the fields where the text search is performed (separated by commas)
 * - multiword_criteria: operator used in criteria for multiple words (AND|OR). By default, "OR"
 * - additional_query_params: additional criteria to be added to the search expression
 * - categories_field: name of the field having the array of categories the product belongs to
 * - related_method: method to get the related products (name|name-reverse|categories|self)
 * - facet_fields: comma separated list of filter fields 
 * - price_range: {start},{end},{gap} Example: 0,1000,200 (group prices from 0 to 1000 in ranges of 200)
 * - http_verb: HTTP verb to use in requests (get|post). By default, "get".
 * - select_method: name of the method to do Select
 * - search_mode: name of the search type
 * 
 * Example:
    host=http://xxxxxxxxx.net:8080/solr;
    id_field=id;
    additional_query_params=in_stock:true AND store_id:1;
    query_fields=fulltext_1_en,fulltext_2_en;
    name_field=attr_sort_name_en;
    price_field=price_0_1;
    related_method=name;
    categories_field=category_ids;
    facet_fields=color_family,size;
    price_range=0,1000,100;
    http_verb=get;
 */

class LetsSyncroLLC_Oct8ne_Helper_Search_Solr extends LetsSyncroLLC_Oct8ne_Helper_Search_Base {

    const PRICE_PARAM_NAME = "price-param";

    public function getEngineName() {
        return "solr";
    }

    public function isValidSearchCriteria($searchTerm) {
        if (is_null($searchTerm) || strlen($searchTerm) == 0) {
            return false;
        }
        return true;
    }

    public function search($storeId, $searchTerm, $searchOrder, $searchDir, $page, $pageSize, &$totalSearchResults, &$attrs_applied, &$attrs_available) {
        if (!$this->checkEngineParams()) {
            Mage::log("[Oct8ne] Invalid Solr engine params");
            return array();
        }

        $this->log("Searching " . $searchTerm);
        $result = $this->executeSolrSearchQuery($searchTerm, NULL, $searchOrder, $searchDir, $page, $pageSize, TRUE /* addFacets */);
        if (is_null($result)) {
            return array();
        }
        $totalSearchResults = $result["response"]["numFound"];

        $allAvailableFilters = $this->getAvailableFilters($result);
        $attrs_applied = $this->getResponseAppliedFilter($allAvailableFilters);
        $attrs_available = $this->getAvailableButNotAppliedFilters($attrs_applied, $allAvailableFilters);
        $productIds = $this->getProductIds($result);
        $this->log("Total {$totalSearchResults}, ids of this page: [" . implode(",", $productIds) . "]");
        return $productIds;
    }

    public function getRelatedProductIds($product, $page, $pageSize) {
        if (!$this->checkEngineParams()) {
            Mage::log("[Oct8ne] Invalid Solr engine params");
            return array();
        }

        $pageSize = 30;
        $relatedMethod = strtolower($this->getEngineParam("related_method"));
        if (!$relatedMethod) {
            $relatedMethod = "name";
        }
        
        if ($relatedMethod == "self") {
            $result = array($product->getId());
            return $result;
        }
        
        $result = NULL;
        if ($relatedMethod == "categories") {
            $categoryIds = $product->getCategoryIds();
            if (!empty($categoryIds)) {
                $result = $this->executeSolrSearchQuery(NULL, $categoryIds, 'score', 'asc', $page, $pageSize, FALSE /* addFacets */);
            }
            $relatedMethod = "name";
        }
        if ($relatedMethod == "name-reverse") {
            $arr = explode(' ', $product->getName());
            $arr = array_reverse($arr);
            $name = implode(' ', $arr);
            $result = $this->executeSolrSearchQuery($name, NULL, 'score', 'desc', $page, $pageSize, FALSE /* addFacets */);
        } else if ($relatedMethod == "name") {
            $name = $product->getName();
            $result = $this->executeSolrSearchQuery($name, NULL, 'score', 'desc', $page, $pageSize, FALSE /* addFacets */);
        }

        if (is_null($result) || !$result || count($result) == 0) {
            $result = array($product->getId());
        }

        $productIds = $this->getProductIds($result);

        return $productIds;
    }

    private function executeSolrSearchQuery($searchTerm, $searchCategories, $searchOrder, $searchDir, $page, $pageSize, $addFacets) {
        if (!function_exists('curl_init')) {
            return NULL;
        }
        $host = $this->normalizeHost($this->getEngineParam("host"));
        if ($page < 1) {
            $page = 1;
        }
        $queryParams = array(
            'rows' => $pageSize,
            'start' => $pageSize * ($page - 1),
            'wt' => 'json',
            'sort' => $this->getSolrSort($searchOrder, $searchDir),
            'fl' => $this->getEngineParam("id_field") . ',' . $this->getEngineParam("name_field") . ',' . $this->getEngineParam("price_field"),
            'facet' => 'true',
            'facet.mincount' => 1
        );

        if ($searchCategories) {
            $categoryFieldName = $this->getEngineParam("categories_field");
            if ($categoryFieldName) {
                // Format: q=category_ids:(2+OR+48)
                $queryParams["q"] = $categoryFieldName . ":(" . implode(" OR ", $searchCategories) . ")";
            }
        }
        
        $selectMethod = $this->getEngineParam("select_method");
        if(!$selectMethod){
            $selectMethod = "select";
        }

        $query = $host . '/'.$selectMethod.'?' . http_build_query($queryParams);

        // Set up query expression
        $queryFields = $this->getEngineParam("query_fields");
        if(! $queryFields) {
            $queryFields = $this->getEngineParam("query_field"); // Retrocompatibility
        }
        $q = trim("" . $this->getEngineParam("additional_query_params"));
        if ($q) {
            $q .= " AND ";
        }
        $searchTerm = trim($searchTerm);
        if ($searchTerm) {
            $searchCriteria = $this->getSearchCriteria($searchTerm, $queryFields); 
            $q .= $searchCriteria;
        }
        $q = str_replace(" ", "%20", $q);
        $query .= "&q=" . $q;

        // Set up facets
        if ($addFacets && $this->getEngineParam("facet_fields")) {
            $query = $this->addAllFacetsToQuery($query);
            $query = $this->addCurrentFacetsToQuery($query);
        }

        // Setup price ranges
        if ($this->getEngineParam("price_range")) {
            $query = $this->addAllPriceRangesToQuery($query);
        }
        
        $this->log("Executing query: {$query}");

        /* Uncomment to debug
            if($searchCriteria) {
                echo "<b>Search criteria:</b><br>" . $searchCriteria . "<br>";
            }
            echo "<b>Complete criteria:</b><br>" . $q . "<br>";
            echo "<b>Query:</b><br>" . $query . "<br>";
            return;
        */

        $request = curl_init($query);
        
        if($this->getEngineParam("http_verb")=="post") {
            curl_setopt($request, CURLOPT_POST, 1);
        }
        curl_setopt($request, CURLOPT_RETURNTRANSFER, 1);

        $userName = $this->getEngineParam("username");
        if ($userName) {
            $password = $this->getEngineParam("password");
            if (is_null($password)) {
                $password = "";
            }
            curl_setopt($request, CURLOPT_USERPWD, "$userName:$password");
        }
        
        $response = curl_exec($request);       
        $err = curl_error($request);        
        if (is_null($err) || $err == "") {
            $result = json_decode($response, true);
            if (is_null($result)) {
                $contentType = curl_getinfo($request, CURLINFO_CONTENT_TYPE);
                $msg = "Solr search is not returning valid JSON data, or result is empty. Content-type returned: " . $contentType;
                Mage::log("[Oct8ne] " . $msg);
                $this->log($msg);
            } else {
                $this->log("Query result received");
            }
        } else {
            $this->log("Search error: " . $err);
            $result = NULL;
        }
        curl_close($request);
        return $result;
    }
    
    private function getSearchCriteria($searchTerm, $queryFields) {
        $fieldsArray = explode(",", $queryFields);
        if(empty($fieldsArray) || !$queryFields) {
            return $searchTerm;
        }
        $result = "(";
        $first = true;
        foreach($fieldsArray as $queryField) {
            if (!$first) {
                $result .= " OR ";
            }
            if ($searchTerm[0] == '"' && $searchTerm[strlen($searchTerm) - 1] == '"') {
                $result .= ("(" . $queryField . ":" . $searchTerm . ")");
            } else {
                $result .= $this->generateQueryExpression($queryField, $searchTerm);
            }
            $first = false;
        }
        $result .= ")";
        return $result;
    }
    
    // Splits the search terms and generates an "OR" search expression
    // Example: "hello world" -> "(text:hello OR text:world)"
    private function generateQueryExpression($queryField, $searchTerm) {
        $words = explode(" ", $searchTerm);
        if(empty($words)) {
            return "";
        }
        $result = "(";
        $first = true;
        
        $searchMode = $this->getEngineParam("search_mode");
        
        $multiwordCriteria = $this->getEngineParam("multiword_criteria");
        if(!$multiwordCriteria) {
            $multiwordCriteria = "OR";
        }
        foreach ($words as $word) {
            $word = trim($word);
            if ($word) {
                if (!$first) {                    
                    $result .=" " . $multiwordCriteria ." ";                    
                }
                if(!$searchMode){
                    $result .= $queryField . ":" . $word . "";
                }              
                else if($searchMode == "CONTAINS"){
                    $result .= $queryField . ":*" . $word . "*";        
                }
                
                $first = false;
            }
        }
        $result .= ")";
        return $result;
    }

    // Add all facets to the query, so 
    // the results will include these facets
    private function addAllFacetsToQuery($query) {
        $facetFieldsPrefix = $this->getFacetFieldsPrefix();
        $facetFields = $this->getEngineParam("facet_fields");
        if ($facetFields) {
            $facetFieldsArray = explode(',', $facetFields);
            foreach ($facetFieldsArray as $facet) {
                $query .= "&facet.field=" . $facetFieldsPrefix . trim($facet);
            }
        }
        return $query;
    }

    // Add the current specified filters to the Solr query
    // using the fq (facet query) parameter
    private function addCurrentFacetsToQuery($query) {

        $appliedFilters = $this->getAppliedFilters();
        $facetFieldsPrefix = $this->getFacetFieldsPrefix();
        $priceRangeConfig = $this->getPriceRangeConfiguration();
        if ($appliedFilters) {
            foreach ($appliedFilters as $param => $value) {
                if ($param != self::PRICE_PARAM_NAME) { // Price is a special facet and must be handled differently
                    $query .= '&fq=' . $facetFieldsPrefix . $param . ':' . urlencode($value);
                } else {
                    $min = $value;
                    $max = $value + $priceRangeConfig["gap"];
                    $query .= '&fq=' . $this->getEngineParam("price_field") . ":[$min%20TO%20$max]";
                }
            }
        }
        return $query;
    }

    // Adds all price ranges to the query as facet range
    // so the results will include the number of products per each range
    private function addAllPriceRangesToQuery($query) {
        $priceField = $this->getEngineParam("price_field");
        $rangeConfig = $this->getPriceRangeConfiguration();
        if ($rangeConfig) {
            $start = $rangeConfig["start"];
            $end = $rangeConfig["end"];
            $gap = $rangeConfig["gap"];
            $query .= "&facet.range=$priceField&f.$priceField.facet.range.start=$start&f.$priceField.facet.range.end=$end&f.$priceField.facet.range.gap=$gap";
        }
        return $query;
    }

    private function getProductIds($result) {
        $productIds = array();
        $idField = $this->getEngineParam('id_field');
        if (!$result || !isset($result["response"]) || !isset($result["response"]["docs"]))
            return $productIds;

        foreach ($result["response"]["docs"] as $product) {
            $productIds[] = $product[$idField];
        }

        // Uncomment for local testing
        // $productIds = array('906', '875', '874', '554', '553', '552', '551', '549', '399', '398');

        return $productIds;
    }

    private function normalizeHost($host) {
        $host = rtrim($host, '/');
        if (!$this->startsWith($host, 'http://') && !!$this->startsWith($host, 'https://')) {
            $host = 'http://' . $host;
        }
        return $host;
    }

    private function startsWith($str, $subStr) {
        $length = strlen($subStr);
        return (substr($str, 0, $length) === $subStr);
    }

    private function getSolrSort($searchOrder, $searchDir) {
        $result = "";
        $relevanceField = $this->getEngineParam("relevance_field");
        switch ($searchOrder) {
            case "score":
                $result .= "score";
                break;
            case "relevance":                
                if($relevanceField){
                     $result .= $relevanceField;
                }else{
                     $result .= "score";
                }         
                break;
            case "price":
                $result .= $this->getEngineParam("price_field");
                break;
            case "name":
                $result .= $this->getEngineParam("name_field");
        }
        if($relevanceField == "top_ventas"){
            $result .= ($searchDir == "desc" ? " asc" : " desc" );
        }else{
            $result .= ($searchDir == "desc" ? " desc" : " asc" );
        }
        
        return $result;
    }

    // Gets all the available filters from the search result
    // using facet_fields and facet_ranges
    private function getAvailableFilters($result) {
        $filters = array();
        $facetCounts = $result["facet_counts"];
        if (!$facetCounts)
            return $filters;

        $facetFieldsPrefix = $this->getFacetFieldsPrefix();

        // First, work with faceted fields
        foreach ($facetCounts["facet_fields"] as $facetName => $values) {
            $facetName = str_replace($facetFieldsPrefix, '', $facetName);

            $attribute = Mage::getSingleton('eav/config')->getAttribute(Mage_Catalog_Model_Product::ENTITY, $facetName);
            $facetLabel = $attribute->getFrontendLabel();
            $optionsDictionary = array();
            if ($attribute->usesSource()) {
                $attrOptions = $attribute->getSource()->getAllOptions(false);
                foreach ($attrOptions as $opt) {
                    $optionsDictionary[$opt["value"]] = $opt["label"];
                }
            }

            $options = array();
            $i = 0;
            while ($i < count($values)) {
                $optionName = $values[$i];
                $optionLabel = isset($optionsDictionary[$optionName]) ? $optionsDictionary[$optionName] : $optionName;
                if ($optionLabel) {
                    $count = $values[$i + 1];
                    if ($count > 0) {
                        $options[] = $this->createFilterOption($optionLabel, $optionName, $count);
                    }
                }
                $i+=2;
            }
            if (count($options) > 0) { // Only add filters with 1 or more options
                $filters[] = $this->createFilterInfo($facetName, $facetLabel, $options);
            }
        }

        // Second, go for price ranges if present
        $ranges = $facetCounts["facet_ranges"];
        if (!$ranges)
            return $filters;
        $priceField = $this->getEngineParam("price_field");
        $priceRanges = $ranges[$priceField];
        if (!$priceRanges)
            return $filters;

        $ranges = $priceRanges["counts"];
        if (!$ranges)
            return $filters;

        $rangeConfig = $this->getPriceRangeConfiguration();
        if (!$rangeConfig)
            return $filters;

        $gap = $rangeConfig["gap"];

        $options = array();
        $i = 0;
        $valueCount = count($ranges);
        while ($i < $valueCount) {
            $priceStart = round($ranges[$i]);
            $count = $ranges[$i + 1];
            if ($count > 0) {
                $options[] = $this->createFilterOption($priceStart . " - " . ($priceStart + $gap), $priceStart, $count);
            }
            $i+=2;
        }
        if (count($options) > 0) {
            $desc = $this->__('Price');
            $filters[] = $this->createFilterInfo(self::PRICE_PARAM_NAME, $desc, $options);
        }
        return $filters;
    }

    // Determines what available filters have been applied in the current request
    private function getResponseAppliedFilter($availableFilters) {
        $filters = array();
        $request = $this->getRequest();
        $config = $this->getPriceRangeConfiguration();
        $gap = $config["gap"];

        foreach ($availableFilters as $availableFilter) {

            $parameterName = $availableFilter["param"];
            $parameterValue = $request->getParam($parameterName);

            if (!is_null($parameterValue) && trim($parameterValue)!="") {

                if ($parameterName == self::PRICE_PARAM_NAME) {
                    $filters[] = $this->createAppliedFilter(
                            self::PRICE_PARAM_NAME, $this->__('Price'), $parameterValue, $parameterValue . " - " . ($parameterValue + $gap)
                    );
                } else {
                    $product = Mage::getModel('catalog/product')->setData($parameterName, $parameterValue);
                    $valueLabel = $product->getAttributeText($parameterName);
                    $filters[] = $this->createAppliedFilter(
                            $availableFilter["param"], $availableFilter["paramLabel"], $parameterValue, $valueLabel
                    );
                }
            }
        }
        return $filters;
    }

    private function checkEngineParams() {
        $valid = $this->check("host");
        $valid = $valid && $this->check("id_field");
        //$valid = $valid && ($this->check("query_fields") || $this->check("query_field")); // Retrocompatible
        $valid = $valid && $this->check("name_field");
        $valid = $valid && $this->check("price_field");
        $valid = $valid && $this->check("categories_field");
        return $valid;
    }

    private function check($paramName) {
        if (!$this->getEngineParam($paramName)) {
            $this->log("Error: param $paramName is not defined");
            return FALSE;
        }
        return TRUE;
    }

    private function getFacetFieldsPrefix() {
        $facetFieldsPrefix = $this->getEngineParam("facet_fields_prefix");
        return $facetFieldsPrefix ? $facetFieldsPrefix : "attr_nav_select_";
    }

    private function getPriceRangeConfiguration() {
        $ranges = $this->getEngineParam("price_range");
        $rangeValues = explode(",", $ranges);
        $start = ($rangeValues && count($rangeValues) > 0) ? $rangeValues[0] : 0;
        $end = ($rangeValues && count($rangeValues) > 1) ? $rangeValues[1] : 9999999;
        $gap = ($rangeValues && count($rangeValues) > 2) ? $rangeValues[2] : 100;
        $result = array(
            "start" => $start,
            "end" => $end,
            "gap" => $gap
        );
        return $result;
    }

}
