<?php

namespace Oct8ne\Oct8ne\Helper;

use \Magento\Framework\App\Helper\AbstractHelper;
use \Magento\Framework\App\Helper\Context;
use \Magento\Catalog\Api\ProductRepositoryInterfaceFactory;
use \Magento\Framework\App\Config\ScopeConfigInterface;
use \Magento\Framework\View\Asset\Repository;
use \Magento\Tax\Api\TaxCalculationInterface;
use \Magento\Framework\Pricing\PriceCurrencyInterface as CurrencyHelper;
use \Magento\Catalog\Block\Product\ListProduct;
use \Magento\Framework\Data\Form\FormKey;
use \Magento\Catalog\Model\Product\Option;
use \Magento\Store\Model\StoreManagerInterface;
use \Magento\Catalog\Helper\Data;

class ProductSummaryHelper extends AbstractHelper
{
    /**
     * @var ProductRepositoryInterfaceFactory
     */
    protected $_productRepository;

    /**
     * @var Context
     */
    protected $_context;

    /**
     * @var Repository
     */
    protected $_assetRepo;

    /**
     * @var TaxCalculationInterface
     */
    protected $_taxCalcualtion;

    /**
     * @var CurrencyHelper
     */
    protected $_currencyHelper;

    /**
     * @var ListProduct
     */
    protected $_listProduct;

    /**
     * @var FormKey
     */
    protected $_formKey;

    /**
     * @var Option
     */
    protected $_option;

    /**
     * @var StoreManagerInterface
     */
    protected $_storeManager;

    /**
     * @var Data
     */
    protected $_taxHelper;

    /**
     * @var ScopeConfigInterface
     */
    protected $_scopeConfig;

    /**
     * Constructor
     *
     * @param Context $context
     * @param StoreManagerInterface $storeManager
     * @param ProductRepositoryInterfaceFactory $productRepository
     * @param Repository $assetRepo
     * @param TaxCalculationInterface $taxCalculation
     * @param CurrencyHelper $currencyHelper
     * @param ListProduct $listproduct
     * @param FormKey $formKey
     * @param Option $option
     * @param Data $taxHelper
     * @param ScopeConfigInterface $scopeConfig
     */
    public function __construct(
        Context $context,
        StoreManagerInterface $storeManager,
        ProductRepositoryInterfaceFactory $productRepository,
        Repository $assetRepo,
        TaxCalculationInterface $taxCalculation,
        CurrencyHelper $currencyHelper,
        ListProduct $listproduct,
        FormKey $formKey,
        Option $option,
        Data $taxHelper,
        ScopeConfigInterface $scopeConfig
    ) {
        parent::__construct($context);

        $this->_productRepository = $productRepository;
        $this->_context = $context;
        $this->_assetRepo = $assetRepo;
        $this->_taxCalcualtion = $taxCalculation;
        $this->_currencyHelper = $currencyHelper;
        $this->_listProduct = $listproduct;
        $this->_formKey = $formKey;
        $this->_option = $option;
        $this->_storeManager = $storeManager;
        $this->_taxHelper = $taxHelper;
        $this->_scopeConfig = $scopeConfig;
    }

    /**
     * Obtiene informacion básica sobre el producto
     *
     * @param int|string $id
     * @param array $oct8ne_context
     * @param bool $extended para especificar si es productInfo o no
     * @return array
     */
    public function getProductSummary($id, $oct8ne_context, $extended = false)
    {
        $result = [];
        try {
            $medias = [];
            $img_folder = $this->_storeManager->getStore()->getBaseUrl(\Magento\Framework\UrlInterface::URL_TYPE_MEDIA)
                . 'catalog/product';

            $storeId = $this->_context->getRequest()->getParam('store', $this->_storeManager->getStore()->getId());
            $searchBy = $this->_scopeConfig->getValue("Oct8ne/user/search_by/" . $storeId, "default", $storeId);

            if ($searchBy == "sku") {
                $product = $this->_productRepository->create()->get($id, false, $oct8ne_context["context_shop"]);
            } else {
                $product = $this->_productRepository->create()->getById($id, false, $oct8ne_context["context_shop"]);
            }

            $rate = $this->getRateValue($product);
            $prices = $this->getPricesWWTaxes($product, $rate);
            $prices = $this->formatPrices($prices, $oct8ne_context);

            $currencyCode = !empty($oct8ne_context["context_currency"])
                ? $oct8ne_context["context_currency"]
                : $oct8ne_context["current_currency"];

            $result["internalId"] = $id;
            $result["title"] = $product->getName();
            $result["formattedPrice"] = $this->_currencyHelper->format(
                $prices["final_incl"],
                false,
                2,
                null,
                $currencyCode
            );
            $result["formattedPrevPrice"] = $this->_currencyHelper->format(
                $prices["incl"],
                false,
                2,
                null,
                $currencyCode
            );
            $result["productUrl"] = $this->cleanProtocol($product->getUrlModel()->getUrl($product));

            $tierPrices = $product->getTierPrice();
            if (!empty($tierPrices)) {
                $result["formattedPrice"] = null;
                $result["formattedPrevPrice"] = null;
            }
            
            if (!empty($product->getImage())) {
                $image = $product->getImage();
                if (substr($image, 0, 1) !== "/") {
                    $image = "/" . $image;
                }
                $result["thumbnail"] = $this->cleanProtocol($img_folder . $image);
            } else {
                $medias = $this->getMediaImages($product, $img_folder, $medias);

                if (empty($medias)) {
                    $result["thumbnail"] = $this->cleanProtocol(
                        $this->_assetRepo->getUrl("Oct8ne_Oct8ne::images/demo.jpg")
                    );
                } else {
                    $result["thumbnail"] = $medias[0]["url"];
                }
            }

            if ($extended) {
                $result["description"] = $product->getDescription();
                $aux = $this->cleanProtocol(
                    $this->_listProduct->getAddToCartUrl($product)
                    . "?form_key=" . $this->_formKey->getFormKey() . "&qty=1"
                );
                $result["addToCartUrl"] = $aux;

                $typeId = $product->getTypeId();
                $hasRequiredOptions = $product->getTypeInstance()->hasRequiredOptions($product);
                $result["useProductUrl"] = "true";
                if ($typeId != 'configurable' && $typeId != 'grouped' && $typeId != 'bundle' && !$hasRequiredOptions) {
                    $result["useProductUrl"] = "false";
                }

                $medias = $this->getMediaImages($product, $img_folder, $medias);
                $result["medias"] = $medias;
            }

            return $result;
        } catch (\Magento\Framework\Exception\NoSuchEntityException $exception) {
            return $result;
        }
    }

    /**
     * Get Media Images
     *
     * @param \Magento\Catalog\Model\Product $product
     * @param string $img_folder
     * @param array $medias
     * @return array
     */
    private function getMediaImages($product, $img_folder, $medias)
    {
        $medias_aux = $product->getMediaGalleryEntries();

        if ($medias_aux) {
            foreach ($medias_aux as $media) {
                if ($media->getMediaType() == "image") {
                    $urlImage = $media->getFile();
                    if (substr($urlImage, 0, 1) !== "/") {
                        $mediaUrl = $this->cleanProtocol($img_folder . "/" . $urlImage);
                    } else {
                        $mediaUrl = $this->cleanProtocol($img_folder . $urlImage);
                    }

                    $existe = false;
                    foreach ($medias as $item) {
                        if ($item['url'] === $mediaUrl) {
                            $existe = true;
                            break;
                        }
                    }

                    if (!$existe) {
                        $medias[] = ["url" => $mediaUrl];
                    }
                }
            }
        }
        return $medias;
    }

    /**
     * Obtiene si hay impuestos o no
     *
     * @param \Magento\Catalog\Model\Product $product
     * @return float|int
     */
    public function getRateValue($product)
    {
        $rate = 0;
        $taxAttribute = $product->getCustomAttribute('tax_class_id');

        if ($taxAttribute) {
            $productRateId = $taxAttribute->getValue();
            if ($productRateId) {
                $rate = $this->_taxCalcualtion->getCalculatedRate($productRateId);
            }
        }

        return $rate;
    }

    /**
     * Obtiene los precios con y sin descuento
     *
     * @param \Magento\Catalog\Model\Product $product
     * @param float $rate
     * @return array
     */
    public function getPricesWWTaxes($product, $rate)
    {
        $prodyctType = $product->getTypeId();

        switch ($prodyctType) {
            case "bundle":
                $regular_price = $product->getPriceInfo()->getPrice('regular_price')->getAmount()->getValue();
                $final_price = $product->getPriceInfo()->getPrice('final_price')->getValue();
                break;

            case "grouped":
            case "configurable":
                $final_price = $product->getPriceInfo()->getPrice('final_price')->getValue();
                $regular_price = $final_price;
                break;

            default:
                $regular_price = $product->getPrice();
                $final_price = $product->getFinalPrice();
                break;
        }

        $configPath = 'tax/calculation/price_includes_tax';
        $priceIncludesTax = (int)$this->_scopeConfig->getValue(
            $configPath,
            \Magento\Store\Model\ScopeInterface::SCOPE_STORE
        );

        if ($priceIncludesTax === 1) {
            $priceExcludingTax = $regular_price / (1 + ($rate / 100));
            $finalPriceExcludingTax = $final_price / (1 + ($rate / 100));
            $priceIncludingTax = $regular_price;
            $finalPriceIncludingTax = $final_price;
        } else {
            $priceExcludingTax = $regular_price;
            $finalPriceExcludingTax = $final_price;
            $priceIncludingTax = $this->_taxHelper->getTaxPrice($product, $regular_price, true);
            $finalPriceIncludingTax = $this->_taxHelper->getTaxPrice($product, $final_price, true);
        }

        return [
            'incl' => $priceIncludingTax,
            'excl' => $priceExcludingTax,
            'final_incl' => $finalPriceIncludingTax,
            'final_excl' => $finalPriceExcludingTax
        ];
    }

    /**
     * Convierte los precios a la moneda requerida
     *
     * @param array $prices
     * @param array $oct8ne_context
     * @return array
     */
    public function formatPrices($prices, $oct8ne_context)
    {
        $contextCurrency = $oct8ne_context["context_currency"] ?? null;
        $currentCurrency = $oct8ne_context["current_currency"] ?? null;

        if (!empty($contextCurrency) && $contextCurrency != $currentCurrency) {
            foreach ($prices as $key => $value) {
                $prices[$key] = $this->_currencyHelper->convertAndRound($value, null, $contextCurrency, 2);
            }
        }

        return $prices;
    }

    /**
     * Quita el protocolo
     *
     * @param string $string
     * @return string
     */
    public function cleanProtocol($string)
    {
        $aux = str_replace("https:", "", $string);
        $aux = str_replace("http:", "", $aux);
        return $aux;
    }
}
