<?php
/**
 * Created by PhpStorm.
 * User: migue
 * Date: 20/06/2017
 * Time: 19:18
 */

namespace Oct8ne\Oct8ne\Helper\Search;

use \Magento\CatalogSearch\Helper\Data as CatalogSearch;
use \Magento\Framework\HTTP\Client\Curl;
use Magento\Framework\App\Config\ScopeConfigInterface;
use \Magento\Store\Model\StoreManagerInterface;
use \Magento\Framework\App\Helper\Context;
use \Magento\Framework\App\RequestInterface;



/**
 * Class Doofinder
 * @package Oct8ne\Oct8ne\Helper\Search
 */
class Doofinder extends Base {

    const ACCOUNT_CONFIG = 'doofinder_config_config/doofinder_account';
    const SEARCH_ENGINE_CONFIG = 'doofinder_config_config/doofinder_search_engine';

    protected $_catalogSearchHelper;
    protected $_curl;
    protected $_storageManager;
    protected $_scopeconfig;
    protected $_request;


    const DEFAULT_TIMEOUT = 10000;
    const DEFAULT_RPP = 10;
    const DEFAULT_PARAMS_PREFIX = 'dfParam_';
    const DEFAULT_API_VERSION = '6';
    const DEFAULT_BASE_ADDRESS = 'https://%s-search.doofinder.com';

    private $api_key = null; // Authentication
    private $zone = null;
    private $hashid = null;  // ID of the Search Engine

    private $apiVersion = null;
    private $baseAddress = null; // to override default api address
    private $extraHeaders = array(); // to add custom headers
    private $query = null;
    private $search_options = array(); // Assoc. array of request parameters

    private $page = 1;          // Current "page" of results
    private $queryName = null;  // Last successful query name
    private $lastQuery = null;  // Last successful query made
    private $total = 0;      // Total number of results
    private $maxScore = null;
    private $paramsPrefix = self::DEFAULT_PARAMS_PREFIX;
    private $serializationArray = null;
    private $queryParameter = 'query';
    private $allowedParameters = array('page', 'rpp', 'timeout', 'types', 'filter', 'query_name', 'transformer', 'sort', 'exclude'); // Valid parameters
    private $sessionId = null; // for stats stuff

    const SUCCESS = 'success';      // everything ok
    const NOTFOUND = 'notfound';    // no account with the provided hashid found
    const EXHAUSTED = 'exhausted';  // the account has reached its query limit

    private $properties = array();
    private $results = array();
    private $facets = array();
    private $filter = array();

    public $status = null;


    public function __construct(Context $context, CatalogSearch $catalogSearchHelper, Curl $curl, StoreManagerInterface $storeManager, RequestInterface $request) {

        $this->_catalogSearchHelper = $catalogSearchHelper;
        $this->_storageManager = $storeManager;
        $this->_curl = $curl;
        $this->_scopeconfig = $context->getScopeConfig();
        $this->_request = $request;

    }

    public function getEngineName()
    {
        return "Doofinder";
    }

    public  function isValidSearchData($searchTerm, $storeI)
    {
        if (is_null($searchTerm) || strlen($searchTerm) == 0) {
            return false;
        }
        $helper = $this->_catalogSearchHelper;
        $len = strlen($searchTerm);
        if ($len < $helper->getMinQueryLength() || $len > $helper->getMaxQueryLength()) {
            return false;
        }
        return true;
    }

    public  function search($storeId, $searchTerm, $searchOrder, $searchDir, $page, $pageSize, $searchBy, &$totalSearchResults, &$attrs_applied, &$attrs_available)
    {

        
        $hashId = $this->getHashId($storeId);
        $key_api = $this->getApiKey($storeId); 
        $results_ = array();

        $_options["rpp"] = $pageSize;


        try {
            $this->init($hashId, $key_api);
            $this->query($searchTerm, $page, $_options);
            $totalSearchResults = $this->total;
            foreach ($this->results as $result) {
                $results_[] = (int)$result["id"];
            }

        }catch (\Exception $e) {

        }

        return $results_;
    }

    public function init($hashid, $api_key, $fromParams = false, $init_options = array()) {
        $zone_key_array = explode('-', $api_key);

        if (2 === count($zone_key_array)) {
            $this->api_key = $zone_key_array[1];
            $this->zone = $zone_key_array[0];
        } else {
            throw new \Exception("API Key is no properly set.");
        }

        if (array_key_exists('prefix', $init_options)) {
            $this->paramsPrefix = $init_options['prefix'];
        }

        $this->allowedParameters = array_map(array($this, 'addPrefix'), $this->allowedParameters);

        if (array_key_exists('queryParameter', $init_options)) {
            $this->queryParameter = $init_options['queryParameter'];
        } else {
            $this->queryParameter = $this->paramsPrefix.$this->queryParameter;
        }

        $this->setApiVersion(self::DEFAULT_API_VERSION);

        if (array_key_exists('restrictedRequest', $init_options)) {
            switch(strtolower($init_options['restrictedRequest'])) {
                case 'get':
                    $this->serializationArray = $this->_request->getParams();
                    break;
                case 'post':
                    $this->serializationArray = $this->_request->getPost();
                    break;
                default:
                    throw new \Exception("Wrong initialization value for 'restrictedRequest'");
            }
        } else {
            $this->serializationArray = $this->_request;
        }

        if (!preg_match('/^[0-9a-f]{32}$/i', $hashid)) {
            throw new \Exception("Wrong hashid");
        }

        $this->hashid = $hashid;

        if ($fromParams) {
            $this->fromQuerystring();
        }
    }

    public function getEndpointURL($entryPoint = '_search', $params = array()) {
        $enpointAddress = sprintf("%s/%s/%s/%s", $this->getServerAddress(), $this->getApiVersion(), $this->hashid, $entryPoint);
        return sprintf("%s?%s", $enpointAddress, http_build_query($this->sanitize($params), '', '&'));
    }

    private function addPrefix($value) {
        return $this->paramsPrefix.$value;
    }

    private function getRequestHeaders(){
        $headers = array();
        $headers[] = 'Expect:'; // Fixes HTTP/1.1 "417 Expectation Failed" Error
        if ($this->authenticationHeader !== false) {
            $headers[] = sprintf("%s: %s", $this->authenticationHeader, $this->api_key);
        }
        foreach($this->extraHeaders as $name => $value){
            $headers[] = sprintf("%s: %s", $name, $value);
        }
        return $headers;
    }

    private function apiCall($entryPoint = '_search', $params = array()){
        $params['hashid'] = $this->hashid;

        
        $this->_curl->setOption(CURLOPT_CUSTOMREQUEST, 'GET');
        $this->_curl->setOption(CURLOPT_HEADER, false);
        $this->_curl->setOption(CURLOPT_RETURNTRANSFER, true);
        $this->_curl->setOption(CURLOPT_HTTPHEADER, $this->getRequestHeaders());
        $this->_curl->get($this->getEndpointURL($entryPoint, $params));
        $response = $this->_curl->getBody();
        $statusCode = $this->_curl->getStatus();

        /*$session = curl_init($this->getEndpointURL($entryPoint, $params));

        // Configure cURL to return response but not headers
        curl_setopt($session, CURLOPT_CUSTOMREQUEST, 'GET');
        curl_setopt($session, CURLOPT_HEADER, false);
        curl_setopt($session, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($session, CURLOPT_HTTPHEADER, $this->getRequestHeaders());

        $response = curl_exec($session);
        $statusCode = curl_getinfo($session, CURLINFO_HTTP_CODE);

        curl_close($session);*/

        if (floor($statusCode / 100) == 2) {
            return $response;
        }

        throw new \Exception($statusCode.' - '.$response, $statusCode);
    }

    public function getOptions() {
        return $this->apiCall('options/'.$this->hashid);
    }


    public function query($query = null, $page = null, $options = array()) {
        if ($query) {
            $this->search_options['query'] = $query;
        }

        if ($page) {
            $this->search_options['page'] = intval($page);
        }

        foreach ($options as $optionName => $optionValue) {
            $this->search_options[$optionName] = $options[$optionName];
        }

        $params = $this->search_options;

        // translate filters
        if (!empty($params['filter']))
        {
            foreach($params['filter'] as $filterName => $filterValue){
                $params['filter'][$filterName] = $this->updateFilter($filterValue);
            }
        }

        // translate excludes
        if (!empty($params['exclude']))
        {
            foreach($params['exclude'] as $excludeName => $excludeValue){
                $params['exclude'][$excludeName] = $this->updateFilter($excludeValue);
            }
        }

        // no query? then match all documents
        if (!$this->optionExists('query') || !trim($this->search_options['query'])){
            $params['query_name'] = 'match_all';
        }

        // if filters without query_name, pre-query first to obtain it.
        if (empty($params['query_name']) && !empty($params['filter']))
        {
            $filter = $params['filter'];
            unset($params['filter']);
            $dfResults = $this->processResult($this->apiCall('_search', $params));
            $params['query_name'] = $dfResults->getProperty('query_name');
            $params['filter'] = $filter;
        }



        $response = $this->apiCall('_search', $params);

        $this->processResult($response);
        $this->page = $this->getProperty('page');
        $this->total = $this->getProperty('total');
        $this->search_options['query'] = $this->getProperty('query');
        $this->maxScore = $this->getProperty('max_score');
        $this->queryName = $this->getProperty('query_name');
        $this->lastQuery = $this->getProperty('query');

        return $this->results;
    }


    public function hasNext(){
        return $this->page * $this->getRpp() < $this->total;
    }


    public function hasPrev(){
        return ($this->page - 1) * $this->getRpp() > 0;
    }


    public function getPage(){
        return $this->page;
    }

    public function setExclude($excludeName, $exclude){
        $this->search_options['exclude'][$excludeName] = (array) $exclude;
    }


    public function getExclude($excludeName){
        if (isset($this->search_options['exclude'][$excludeName])) {
            return $this->search_options['exclude'][$excludeName];
        }

        return false;
    }


    public function getExcludes() {
        if (isset($this->search_options['exclude'])) {
            return $this->search_options['exclude'];
        }

        return array();
    }


    public function setFilter($filterName, $filter){
        $this->search_options['filter'][$filterName] = (array) $filter;
    }


    public function getFilter($filterName){
        if (isset($this->search_options['filter'][$filterName])) {
            return $this->search_options['filter'][$filterName];
        }

        return false;
    }


    public function getFilters() {
        if (isset($this->search_options['filter'])) {
            return $this->search_options['filter'];
        }

        return array();
    }


    public function addTerm($filterName, $term) {
        $this->search_options['filter'][$filterName][] = $term;
    }


    public function removeTerm($filterName, $term) {
        if (isset($this->search_options['filter'][$filterName])) {
            $idx = array_search($term, $this->search_options['filter'][$filterName]);
            if ($idx !== false) {
                array_splice($this->search_options['filter'][$filterName], $idx, 1);
            }
        }
    }


    public function setRange($filterName, $from = null, $to = null) {
        if (!is_null($from))
        {
            $this->search_options['filter'][$filterName]['from'] = $from;
        }
        if (!is_null($to))
        {
            $this->search_options['filter'][$filterName]['to'] = $to;
        }
    }


    public function addSort($sortName, $direction) {
        $this->search_options['sort'][] = array($sortName => $direction);
    }


    public function toQuerystring($page = null){
        $toParams = array();

        foreach ($this->search_options as $paramName => $paramValue) {
            if ($paramName == 'query') {
                $toParams[$this->queryParameter] = $paramValue;
            } else {
                $toParams[$this->paramsPrefix.$paramName] = $paramValue;
            }
        }

        if (!is_null($page)) {
            $toParams[$this->paramsPrefix.'page'] = $page;
        }

        return http_build_query($toParams, '', '&');
    }


    public function fromQuerystring(){
        $filteredParams = array_filter(array_keys($this->serializationArray),
            array($this, 'belongsToDoofinder'));

        foreach ($filteredParams as $param) {
            if ($param == $this->queryParameter) {
                $key = 'query';
            } else {
                $key = substr($param, strlen($this->paramsPrefix));
            }

            $this->search_options[$key] = $this->serializationArray[$param];
        }
    }


    private function updateFilter($filter) {
        $new_filter = array();

        foreach($filter as $key => $value) {
            if ($key === 'from') {
                $new_filter['gte'] = $value;
            } else if ($key === 'to') {
                $new_filter['lte'] = $value;
            } else {
                $new_filter[$key] = $value;
            }
        }

        return $new_filter;
    }


    private function sanitize($params) {
        $result = array();

        foreach ($params as $name => $value) {
            if (is_array($value)) {
                $result[$name] = $this->sanitize($value);
            } else if (trim($value)) {
                $result[$name] = $value;
            } else if($value === 0) {
                $result[$name] = $value;
            }
        }

        return $result;
    }


    private function belongsToDoofinder($paramName){

        $pos = strpos($paramName, '[');

        if ($pos !== false) {
            $paramName = substr($paramName, 0, $pos);
        }

        return in_array($paramName, $this->allowedParameters) || $paramName == $this->queryParameter;
    }


    private function optionExists($optionName) {
        return array_key_exists($optionName, $this->search_options);
    }


    public function nextPage() {
        return $this->hasNext() ? $this->query($this->lastQuery, $this->page + 1) : null;
    }


    public function prevPage() {
        return $this->hasPrev() ? $this->query($this->lastQuery, $this->page - 1) : null;
    }


    public function numPages() {
        return ceil($this->total / $this->getRpp());
    }

    public function getRpp() {
        $rpp = $this->optionExists('rpp') ? $this->search_options['rpp'] : null;
        return $rpp ? $rpp : self::DEFAULT_RPP;
    }


    public function setApiVersion($apiVersion) {
        $apiVersion = trim($apiVersion);

        switch (true) {
            case intval($apiVersion) == 4:
                $this->authenticationHeader = 'API Token';
                break;
            case intval($apiVersion) == 5:
                $this->authenticationHeader = 'Authorization';
                break;
            case intval($apiVersion) == 6:
                $this->authenticationHeader = 'Authorization';
                break;
            default:
                throw new \Exception('Wrong API Version');
        }

        $this->apiVersion = $apiVersion;
    }


    public function setPrefix($prefix) {
        $this->paramsPrefix = $prefix;
    }


    private function getFilterType($filter) {
        if (is_array($filter)) {
            if (array_key_exists('from', $filter) || array_key_exists('to', $filter)) {
                return 'numericrange';
            } else {
                return 'term';
            }
        }
        return false;
    }


    private function getHash($refresh = false){
        if(!$this->sessionId || $refresh){
            $time = time();
            $rand = rand();
            $this->sessionId = sha1("$time$rand");
        }
        return $this->sessionId;
    }


    public function cleanSession(){
        $this->sessionId = null;
    }


    public function initSession(){
        return $this->apiCall('stats/init', array('session_id'=>$this->getHash())) == '"OK"';
        return $response == '"OK"';
    }


    public function registerClick($id, $datatype, $query){
        $response = $this->apiCall('stats/click', array('id'=>$id, 'datatype'=>$datatype, 'query'=>$query));
        return $response == '"OK"';
    }


    public function registerCheckout(){
        return $this->apiCall('stats/checkout', array('session_id'=>$this->getHash())) == '"OK"';
    }


    public function registerBannerDisplay($bannerId){
        return $this->apiCall('stats/banner_display', array('banner_id'=>$bannerId)) == '"OK"';
    }


    public function registerBannerClick($bannerId){
        return $this->apiCall('stats/banner_click', array('banner_id'=>$bannerId)) == '"OK"';
    }


    public function registerRedirection($redirectionId, $query, $link){
        return $this->apiCall('stats/redirect', array('redirection_id'=>$redirectionId, 'query'=>$query, 'link'=>$link)) == '"OK"';
    }


    public function setBaseAddress($baseAddress) {
        $baseAddress = trim($baseAddress);
        $this->baseAddress = $baseAddress ? $baseAddress : self::DEFAULT_BASE_ADDRESS;
    }


    public function getApiVersion() {
        return $this->apiVersion;
    }


    public function getServerAddress() {
        return sprintf(
            $this->baseAddress ? $this->baseAddress : self::DEFAULT_BASE_ADDRESS,
            $this->zone
        );
    }


    public function setExtraHeaders($extraHeaders){
        $this->extraHeaders = $extraHeaders;
    }

    /////RESULT

    public function processResult($jsonResponse) {

        $response = json_decode($jsonResponse, true);

        foreach ($response as $key => $value){
            if (!is_array($value)) {
                $this->properties[$key] = $value;
            }
        }

        // Status
        if (isset($this->properties['doofinder_status'])) {
            $this->status = $this->properties['doofinder_status'];
        } else {
            $this->status = self::SUCCESS;
        }

        // Results
        if (isset($response['results'])) {
            $this->results = $response['results'];
        }

        // Redirections
        if (isset($response['redirection'])) {
            $this->properties['redirection'] = $response['redirection'];
        }

        // Banner
        if(isset($response['banner'])) {
            $this->properties['banner'] = $response['banner'];
        }

        // Build a "friendly" filters array
        if (isset($response['filter'])) {
            foreach($response['filter'] as $filterType => $filters) {
                foreach($filters as $filterName => $filterProperties) {
                    $this->filter[$filterName] = $filterProperties;
                }
            }
        }

        // facets
        if (isset($response['facets'])) {
            $this->facets = $response['facets'];
        }

        // mark "selected" true or false according to filters presence
        foreach ($this->facets as $name => $properties) {
            switch (true) {
                case isset($properties['terms']):
                    foreach($properties['terms']['buckets'] as $idx => $bucket) {
                        $this->facets[$name]['terms']['buckets'][$idx]['selected'] = isset($this->filter[$name]) &&
                            in_array($bucket['key'], $this->filter[$name]);
                    }
                    break;
                case isset($properties['range']):
                    foreach(array_keys($properties['range']['buckets']) as $idx) {
                        $this->facets[$name]['range']['buckets'][$idx]['selected_from'] = isset($this->filter[$name]['gte']) ? $this->filter[$name]['gte'] : false;
                        $this->facets[$name]['range']['buckets'][$idx]['selected_to'] = isset($this->filter[$name]['lte']) ? $this->filter[$name]['lte'] : false;
                    }
                    break;
            }
        }
    }


    public function getProperty($name) {
        return array_key_exists($name, $this->properties) ? $this->properties[$name] : null;
    }


    public function getResults(){
        return $this->results;
    }


    public function getFacetsNames(){
        return array_keys($this->facets);
    }


    public function getLegacyFacet($facetName){
        return $this->facets[$facetName];
    }


    public function getFacet($facetName){
        $facetProperties = $this->facets[$facetName];
        switch(true) {
            case isset($facetProperties['terms']):
                return array(
                    "count" => $facetProperties['doc_count'],
                    'terms' => array_map(
                        function($el) {
                            return array(
                                'term' => $el['key'],
                                'count' => $el['doc_count'],
                                'selected' => isset($el['selected'])
                            );
                        },
                        $facetProperties['terms']['buckets'])
                );
            case isset($facetProperties['range']):
                $bucket = $facetProperties['range']['buckets'][0];
                return array(
                    'count' => $bucket['stats']['count'],
                    'from' => $bucket['stats']['min'],
                    'to' => $bucket['stats']['max'],
                    'selected_from' => isset($bucket['selected_from']) ? $bucket['selected_from'] : false,
                    'selected_to' => isset($bucket['selected_to']) ? $bucket['selected_to'] : false
                );
            default:
                break;
        }

    }


    public function getFacets(){
        return $this->facets;
    }


    public function getAppliedFilters(){
        return $this->filter;
    }

    public function isOk() {
        return $this->status == self::SUCCESS;
    }


    //scope
    public function getScopeStore()
    {
       
        return \Magento\Store\Model\ScopeInterface::SCOPE_STORE;
    }

    public function getApiKey($storeId = null)
    {
        return $this->_scopeconfig->getValue('Oct8ne/doofinder/apikey/'.$storeId, "default", $storeId);
    }

    public function getHashId($storeId = null)
    {
        return $this->_scopeconfig->getValue('Oct8ne/doofinder/hashid/'.$storeId, "default", $storeId);

    }

    public function getStoreCode($store = null)
    {
        return $this->_storageManager->getStore($store)->getCode();
    }
}
