<?php
declare(strict_types=1);

namespace Oct8ne\Oct8ne\Observer;

use Magento\Framework\Event\ObserverInterface;
use Magento\Framework\Event\Observer;
use Magento\Framework\HTTP\Client\Curl;
use Magento\Framework\Module\ModuleListInterface;
use Magento\AdminNotification\Model\Inbox;
use Magento\Framework\FlagManager;
use Psr\Log\LoggerInterface;

class CheckUpdateObserver implements ObserverInterface
{
    public const API_URL = 'https://oct8ne.com/plugins/magento_2/update.json';
    public const MODULE_NAME = 'Oct8ne_Oct8ne';
    public const FLAG_CODE = 'oct8ne_update_check_last_run';

    /**
     * @var Curl
     */
    protected $curl;

    /**
     * @var ModuleListInterface
     */
    protected $moduleList;

    /**
     * @var Inbox
     */
    protected $inbox;

    /**
     * @var FlagManager
     */
    protected $flagManager;

    /**
     * @var LoggerInterface
     */
    protected $logger;

    /**
     * Constructor
     *
     * @param Curl $curl
     * @param ModuleListInterface $moduleList
     * @param Inbox $inbox
     * @param FlagManager $flagManager
     * @param LoggerInterface $logger
     */
    public function __construct(
        Curl $curl,
        ModuleListInterface $moduleList,
        Inbox $inbox,
        FlagManager $flagManager,
        LoggerInterface $logger
    ) {
        $this->curl = $curl;
        $this->moduleList = $moduleList;
        $this->inbox = $inbox;
        $this->flagManager = $flagManager;
        $this->logger = $logger;
    }

    /**
     * Execute observer
     *
     * @param Observer $observer
     * @return void
     */
    public function execute(Observer $observer)
    {
        if (!$this->shouldCheckUpdate()) {
             return;
        }

        try {
            $this->updateLastCheckTime();

            $this->curl->setOption(CURLOPT_TIMEOUT, 3);
            $this->curl->get(self::API_URL);
            if ($this->curl->getStatus() !== 200) {
                return;
            }
            $response = $this->curl->getBody();

            if (!$response) {
                return;
            }
            $data = json_decode($response, true);

            if (!is_array($data) || empty($data['version']) || empty($data['url'])) {
                return;
            }
            
            $remoteVersion = $data['version'];
            $currentVersion = $this->getInstalledVersion();
            if (version_compare($remoteVersion, $currentVersion, '>')) {
                $this->addNotification($remoteVersion, $data);
            }

        } catch (\Exception $e) {
            $this->logger->info('Oct8ne Update Check Error: ' . $e->getMessage());
        }
    }

    /**
     * Check if update check should run
     *
     * @return bool
     */
    private function shouldCheckUpdate()
    {
        $lastCheck = $this->flagManager->getFlagData(self::FLAG_CODE);
        if (!$lastCheck) {
            return true;
        }
        return (time() - (int)$lastCheck) > 86400;
    }

    /**
     * Update last check time flag
     *
     * @return void
     */
    private function updateLastCheckTime()
    {
        $this->flagManager->saveFlag(self::FLAG_CODE, time());
    }

    /**
     * Get installed module version
     *
     * @return string
     */
    private function getInstalledVersion()
    {
        $moduleInfo = $this->moduleList->getOne(self::MODULE_NAME);
        return $moduleInfo['setup_version'] ?? '0.0.0';
    }

    /**
     * Add notification to inbox
     *
     * @param string $newVersion
     * @param array $data
     * @return void
     */
    private function addNotification($newVersion, $data)
    {
        $title = "Actualización Oct8ne v$newVersion";
        $description = isset($data['description']) ? $data['description'] : '';
        $url = isset($data['url']) ? $data['url'] : '#';

        $collection = $this->inbox->getCollection();
        $collection->addFieldToFilter('title', $title);
        $collection->addFieldToFilter('is_remove', 0);

        if ($collection->getSize() > 0) {
            return;
        }

        $this->inbox->addNotice($title, $description, $url);
    }
}
