<?php
/**
 * Prestashop module for Oct8ne
 *
 * @author    oct8ne.com
 * @copyright 2025 Oct8ne
 * @license   Commercial license see license.txt *
 *
 * @category  Prestashop
 * @category  Module
 * Support by mail  : support@oct8ne.com
 */
if (!defined('_PS_VERSION_')) {
    exit;
}
require_once dirname(__FILE__) . '/AbstractSearchEngine.php';
require_once _PS_MODULE_DIR_ . 'oct8ne/classes/Oct8neContext.php';

/**
 * Implementacion especifica de EcommFinder para Oct8ne
 * Class OctEcommFinderSearch
 */
class OctEcommFinderSearch extends AbstractSearchEngine
{
    private function getContext()
    {
        if (isset($this->context)) {
            return $this->context;
        }

        return Oct8neContext::get($this);
    }

    public function getEngineInfo()
    {
        $info = [
            'engine' => 'EcommFinderSearch',
            'languages' => [],
        ];

        return $info;
    }

    public function doSearch($language, $search, $page, $pageSize, $orderBy, $dir)
    {
        if (Module::isEnabled('ecommfinder')) {
            $ecommFinder = Module::getInstanceByName('ecommfinder');

            if ($ecommFinder && version_compare($ecommFinder->version, '1.2.0', '>=')) {
                $ecommFinderIsShopConnected = $ecommFinder->isShopConnected();  // bool
                $ecommFinderShopLayer = $ecommFinder->getShopLayer();           // int|null
                $ecommFinderShopReference = $ecommFinder->getShopReference();   // string|null
            } else {
                return [
                    'total' => 0,
                    'result' => [],
                    'error' => 'Version not allowed. Upgrade EcommFinder >= 1.0.2',
                ];
            }
        } else {
            return [
                'total' => 0,
                'result' => [],
                'error' => 'Module EcommFinder not enabled',
            ];
        }

        if (!$ecommFinderIsShopConnected) {
            return [
                'total' => 0,
                'result' => [],
                'error' => 'EcommFinder not Connected',
            ];
        }

        $iso_code = Language::getIsoById($language);

        $currency_code = 'eur'; // valor por defecto

        if ($this->getContext()->currency instanceof Currency) {
            $currency = new Currency($this->getContext()->currency->id);
            $currency_code = strtolower($currency->iso_code);
        }

        $endpoint = 'https://layer{layerId}.ecommfinder.com:1111/search?timestamp=' . time();

        $data = [
            'o' => 'oct8ne',
            'q' => $search,
            'l' => $iso_code,
            'c' => $currency_code,
            's' => $ecommFinderShopReference,
            'p' => $page,
        ];

        $endpoint = str_replace('{layerId}', $ecommFinderShopLayer, $endpoint);

        $curl = curl_init($endpoint);
        $postData = http_build_query($data);
        $domain = $this->getDomain();

        curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($curl, CURLOPT_POST, true);
        curl_setopt($curl, CURLOPT_POSTFIELDS, $postData);
        curl_setopt($curl, CURLOPT_HTTPHEADER, [
            'Content-Type: application/x-www-form-urlencoded',
            'Cache-Control: no-cache',
            'Content-Length: ' . strlen($postData),
            'Origin: ' . $domain,
        ]);

        $output = curl_exec($curl);

        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);

        if ($httpCode != 200) {
            error_log('HTTP error code: ' . $httpCode);
        }

        // Check the return value of curl_exec()
        if ($output === false) {
            throw new Exception(curl_error($curl), curl_errno($curl));
        }
        curl_close($curl);

        if (version_compare(_PS_VERSION_, '8.0', '<')) {
            $result = Tools::jsonDecode($output);
        } else {
            $result = json_decode($output);
        }

        $total = (int) $result->responseInfo->resultsCount;
        $products = [];

        if (is_array($result->searchResults) && count($result->searchResults) > 0) {
            foreach ($result->searchResults as $item) {
                $products[] = [
                    'internalId' => $item->idp,
                    'title' => $item->name,
                    'formattedPrice' => $item->pfin,
                    'formattedPrevPrice' => $item->pori,
                    'productUrl' => $item->url,
                    'thumbnail' => $item->img,
                ];
            }
        }

        return [
            'total' => $total,
            'result' => $products,
        ];
    }

    public function getDomain()
    {
        $base_url = __PS_BASE_URI__;
        $shop_domain = Tools::getShopDomainSsl(true, true);

        return rtrim($shop_domain . $base_url, '/');
    }
}
