<?php
/**
 * Prestashop module for Oct8ne
 *
 * @author    Prestaquality.com
 * @copyright 2016 Prestaquality
 * @license   Commercial license see license.txt *
 *
 * @category  Prestashop
 * @category  Module
 * Support by mail  : info@prestaquality.com
 */
if (!defined('_PS_VERSION_')) {
    exit;
}

class Oct8ne extends Module
{
    private static $EMAIL_CK = 'OCT_EMAIL';

    private static $PASSWORD_CK = 'OCT_PASS';

    private static $API_TOKEN_CK = 'OCT_API_TOKEN';

    private static $LICENSE_ID_CK = 'OCT_LICID';

    public static $SEARCH_ENGINE = 'OCT_SEARCH_ENGINE';

    public static $POSITION_LOAD = 'OCT_POSITION_LOAD';

    public static $URL_IMG_TYPE = 'OCT_URL_IMG_TYPE';

    public static $SERVER_OC_CK = 'OCT_SERVER';

    public static $URL_STATIC_OC_CK = 'OCT_URL_STATIC';

    public static $URL_API_EMPATHYBROKER = 'OCT_URL_API_EMPATHYBROKER';

    public static $ORDER_DETAILS_BY = 'OTC_ORDER_DETAILS_BY';

    public static $CLIENT_EXTRA_SCRIPT = 'OCT_CLIENT_EXTRA_SCRIPT';

    public static $SCRIPT_LOAD_EVENTS = 'OCT_CLIENT_EVENTS';

    public static $SCRIPT_LOAD_TIMER = 'OCT_CLIENT_TIMER';

    public static $URL_API_APISEARCH = 'OCT_URL_API_APISEARCH';

    public static $TENANT_APISEARCH = 'OCT_TENANT_APISEARCH';

    public static $ENDPOINT_APISEARCH = 'OCT_ENDPOINT_APISEARCH';

    public static $APPID_APISEARCH = 'OCT_APPID_APISEARCH';

    public static $INDEXID_APISEARCH = 'OCT_INDEXID_APISEARCH';

    public static $TOKEN_APISEARCH = 'OCT_TOKEN_APISEARCH';

    public static $SITE_APISEARCH = 'OCT_SITE_APISEARCH';

    // public static $LANG_APISEARCH = "OCT_LANG_APISEARCH";
    public static $DEFAULT_ENDPOINT_APISEARCH = 'https://apisearch.cloud/{index_id}/query.json?site={site}&language={language}&page={page}&size={size}&q={query}';

    public static $URL_API_MOTIVESEARCH = 'OCT_URL_API_MOTIVESEARCH';

    public static $TENANT_MOTIVESEARCH = 'OCT_TENANT_MOTIVESEARCH';

    public static $DOOFINDER_API_KEY = 'OCT_DOOFINDER_API_KEY';

    public static $DOOFINDER_HASH_ID = 'OCT_DOOFINDER_HASH_ID';

    public static $DOOFINDER_SEARCH_ZONE = 'OCT_DOOFINDER_SEARCH_ZONE';

    public static $DOOFINDER_QUERY_NAME = 'OCT_DOOFINDER_QUERY_NAME';

    public static $DOOFINDER_INDICES = 'OCT_DOOFINDER_INDICES';

    public static $DOOFINDER_PRODUCTID_PROPERTY = 'OCT_DOOFINDER_PRODUCTID_PROPERTY';

    private static $error_file_name = 'error_log.log';

    private static $class_folder = 'classes/';

    private static $lib_folder = 'lib/';

    /**
     * añade al log la información sobre una excepcion
     *
     * @param Exception $ex
     */
    public function logException($ex)
    {
        $this->logError($ex->getFile() . ':' . $ex->getLine() . ' -> ' . $ex->getMessage());
    }

    private function getContext()
    {
        return $this->context;
    }

    /**
     * Añade la información indicada al log de excepciones
     *
     * @param type $msg
     */
    public function logError($msg)
    {
        // Recogemos el nombre del archivo local y general
        $files_log = [$this->getLocalPath() . self::$error_file_name, _PS_ROOT_DIR_ . '/log/' . date('Ymd') . '_exception.log'];
        $logger = new FileLogger();
        // Guardamos el log
        foreach ($files_log as $file_log) {
            $logger->setFilename($file_log);
            $logger->logError($msg);
        }
    }

    /**
     * Carga la clase solicitada
     *
     * @param type $class_name Nombre de la clase
     *
     * @throws Exception Si el archivo no existe
     */
    public function loadClass($class_name)
    {
        // Componemos el path
        $path = $this->getLocalPath() . self::$class_folder . $class_name . '.php';

        // Comprobamos que exista
        if (!file_exists($path)) {
            throw new Exception('This class can not be loaded: ' . $class_name);
        }
        // La incluimos
        require_once $path;
    }

    /**
     * Carga la libreria solicitada
     *
     * @param type $class_name Nombre de la clase
     *
     * @throws Exception Si el archivo no existe
     */
    public function loadLibrary($class_name, $internal_path = '')
    {
        if (!empty($internal_path)) {
            $internal_path .= '/';
        }
        // Componemos el path
        $path = $this->getLocalPath() . self::$lib_folder . $internal_path . $class_name . '.php';

        // Comprobamos que exista
        if (!file_exists($path)) {
            throw new Exception('This library can not be loaded: ' . $class_name);
        }
        // La incluimos
        require_once $path;
    }

    /**
     * Remove hthp: or https:
     *
     * @param $string
     *
     * @return mixed
     */
    public static function removeHttProtocol($string)
    {
        $string = str_replace('https:', '', $string);
        $string = str_replace('http:', '', $string);

        return $string;
    }

    /**
     * @return string
     */
    public function getTENANTMOTIVESEARCH()
    {
        return self::$TENANT_MOTIVESEARCH;
    }

    /**
     * @return string
     */
    public function getURLMOTIVESEARCH()
    {
        return self::$URL_API_MOTIVESEARCH;
    }

    /**
     * @return string
     */
    public function getENDPOINTAPISEARCH()
    {
        return self::$ENDPOINT_APISEARCH;
    }

    /**
     * @return string
     */
    public function getSITEAPISEARCH()
    {
        return self::$SITE_APISEARCH;
    }

    /**
     * @return string
     */
    public function getTOKENAPISEARCH()
    {
        return self::$TOKEN_APISEARCH;
    }

    /**
     * @return string
     */
    public function getINDEXIDAPISEARCH()
    {
        return self::$INDEXID_APISEARCH;
    }

    /**
     * @return string
     */
    public function getAPPIDAPISEARCH()
    {
        return self::$APPID_APISEARCH;
    }

    /**
     * @return string
     */
    public function getURLEMPATHYBORKER()
    {
        return self::$URL_API_EMPATHYBROKER;
    }

    /**
     * @return string
     */
    public function getDOOFINDERAPIKEY()
    {
        return self::$DOOFINDER_API_KEY;
    }

    /**
     * @return string
     */
    public function getDOOFINDERSEARCHZONE()
    {
        return self::$DOOFINDER_SEARCH_ZONE;
    }

    /**
     * @return string
     */
    public function getDOOFINDERQUERYNAME()
    {
        return self::$DOOFINDER_QUERY_NAME;
    }

    /**
     * @return string
     */
    public function getDOOFINDERINDICES()
    {
        return self::$DOOFINDER_INDICES;
    }

    /**
     * @return string
     */
    public function getDOOFINDERPRODUCTIDPROPERTY()
    {
        return self::$DOOFINDER_PRODUCTID_PROPERTY;
    }

    /**
     * @return string
     */
    public function getDOOFINDERHASHID()
    {
        return self::$DOOFINDER_HASH_ID;
    }

    /**
     * @return string
     */
    public function getTENANTAPISEARCH()
    {
        return self::$TENANT_APISEARCH;
    }

    /**
     * @return string
     */
    public function getURLAPISEARCH()
    {
        return self::$URL_API_APISEARCH;
    }

    /**
     * @return string
     */
    public function getORDERDETAILSBYNAME()
    {
        return self::$ORDER_DETAILS_BY;
    }

    /**
     * @return string
     */
    public function getSEARCHENGINENAME()
    {
        return self::$SEARCH_ENGINE;
    }

    /**
     * @return string
     */
    public function getPOSITIONLOADNAME()
    {
        return self::$POSITION_LOAD;
    }

    /**
     * @return string
     */
    public function getURLIMGTYPENAME()
    {
        return self::$URL_IMG_TYPE;
    }

    /**
     * Oct8ne constructor.
     * Variables de configuracion del modulo
     */
    public function __construct()
    {
        $this->name = 'oct8ne';
        $this->tab = 'front_office_features';
        $this->version = '1.2.8';
        $this->author = 'Oct8ne';
        $this->need_instance = 0;
        $this->module_key = '75e2a2709ae2d368b085faf7c049ac4e';
        $this->ps_versions_compliancy = [
            'min' => '1.4.5.1',
            'max' => '9.0.99',
        ];

        /*
         * Set $this->bootstrap to true if your module is compliant with bootstrap (PrestaShop 1.6)
         */
        $this->bootstrap = true;

        parent::__construct();

        $this->displayName = $this->l('Oct8ne');
        $this->description = $this->l('Oct8ne module connector for Prestashop');

        $this->confirmUninstall = $this->l('All data will be deleted ¿continue?');
    }

    /**
     * Instalacion
     *
     * @param bool $full : completa o parcial
     *
     * @return bool
     */
    public function install($full = true)
    {
        try {
            $correct = parent::install();

            if ($full) {
                $tryhtac = $this->setHtaccessRules();

                if (!$tryhtac) {
                    throw new Exception('htaccess not found');
                }
                // Configuración
                Configuration::updateValue(Oct8ne::$POSITION_LOAD, 1);
                Configuration::updateValue(Oct8ne::$URL_IMG_TYPE, 1);
                Configuration::updateValue(Oct8ne::$SCRIPT_LOAD_EVENTS, 'DISABLED');
                Configuration::updateValue(Oct8ne::$SCRIPT_LOAD_TIMER, 0);

                Configuration::updateValue(
                    Oct8ne::$URL_API_EMPATHYBROKER,
                    'https://api.empathybroker.com/search/v1/query/INSTANCE_ID/search?q='
                );

                Configuration::updateValue(
                    Oct8ne::$URL_API_APISEARCH,
                    'https://search-applications-0.api.motive.co/search'
                );

                Configuration::updateValue(
                    Oct8ne::$URL_API_MOTIVESEARCH,
                    'https://search.api.motive.co/search'
                );
            }
            // Ganchos
            $correct = $correct && $this->registerHook('displayFooter') && $this->registerHook('displayHeader') && $this->registerHook('displayOrderConfirmation');

            return $correct;
        } catch (Exception $ex) {
            $this->logException($ex);

            return false;
        }
    }

    /**
     * Desinstalacion
     *
     * @param bool $full : completa o parcial
     *
     * @return bool
     */
    public function uninstall($full = true)
    {
        if ($full) {
            include dirname(__FILE__) . '/sql/uninstall.php';
            $this->removeHtaccessRules();
            Configuration::deleteByName(self::$EMAIL_CK);
            Configuration::deleteByName(self::$API_TOKEN_CK);
            Configuration::deleteByName(self::$LICENSE_ID_CK);

            Configuration::deleteByName(self::$SEARCH_ENGINE);
            Configuration::deleteByName(self::$POSITION_LOAD);
            Configuration::deleteByName(self::$URL_IMG_TYPE);
            Configuration::deleteByName(self::$URL_API_EMPATHYBROKER);
            Configuration::deleteByName(self::$CLIENT_EXTRA_SCRIPT);
            Configuration::deleteByName(self::$SCRIPT_LOAD_EVENTS);
            Configuration::deleteByName(self::$SCRIPT_LOAD_TIMER);

            Configuration::deleteByName(self::$TENANT_APISEARCH);
            Configuration::deleteByName(self::$TENANT_MOTIVESEARCH);
        }

        return parent::uninstall();
    }

    /**
     * Resetear modulo
     *
     * @return bool
     */
    public function reset()
    {
        if (!$this->uninstall(false)) {
            return false;
        }

        if (!$this->install(false)) {
            return false;
        }

        return true;
    }

    /**
     * Vista configuracion del modulo
     *
     * @return mixed
     */
    public function getContent()
    {
        ContextCore::getContext()->controller->addJS(_PS_MODULE_DIR_ . 'oct8ne/views/js/oct8neadmin.js?v=' . $this->version);

        // Ejecutamos el postprocess
        $this->postProcess();

        $api_key = Configuration::get(self::$API_TOKEN_CK);

        // si no hay api key es que no estamos registrados por lo tanto llamamos a iniciar sesion
        if (empty($api_key)) {
            return $this->renderSettingsForm();
        // si hay api key es que estamos iniciados, mostramos una vista para cerrar sesión
        } else {
            return $this->renderLoggedForm() . $this->renderSearchEngineForm() . $this->renderPositionLoadForm();
        }
    }

    /**
     * Submit Llegada de formularios
     */
    public function postProcess()
    {
        try {
            // si el submit es de tipo login, hacemos las comprobaciones pertinentes
            if ((bool) Tools::isSubmit('login') == true) {
                $email = pSQL(trim(strip_tags(Tools::getValue(self::$EMAIL_CK))));
                $password = pSQL(trim(strip_tags(Tools::getValue(self::$PASSWORD_CK))));
                $check = $this->checkOct8neLinkUp($email, $password);

                if (!empty($email) && !empty($password) && $check != false) {                    // guardamos las variables de configuracion si to.do está correcto
                    Configuration::updateValue(self::$LICENSE_ID_CK, $check['license']);
                    Configuration::updateValue(self::$API_TOKEN_CK, $check['token']);
                    Configuration::updateValue(self::$EMAIL_CK, $email);
                    Configuration::updateValue(self::$SEARCH_ENGINE, 1);
                    Configuration::updateValue(self::$POSITION_LOAD, 1);
                    Configuration::updateValue(self::$URL_IMG_TYPE, 1);

                    Configuration::updateValue(self::$SERVER_OC_CK, $check['server']);
                    Configuration::updateValue(self::$URL_STATIC_OC_CK, $check['urlstatic']);
                } else {
                    $error_msg = $this->l('Cannot login, please check your credentials');
                    $this->context->controller->errors[] = $error_msg;
                }

            // si el submit es de tipo logout, borramos la configuracion
            } elseif ((bool) Tools::isSubmit('logout') == true) {
                Configuration::deleteByName(self::$EMAIL_CK);
                Configuration::deleteByName(self::$API_TOKEN_CK);
                Configuration::deleteByName(self::$LICENSE_ID_CK);
                Configuration::deleteByName(self::$SEARCH_ENGINE);
                Configuration::deleteByName(self::$POSITION_LOAD);
                Configuration::deleteByName(self::$URL_IMG_TYPE);
                Configuration::deleteByName(self::$SERVER_OC_CK);
                Configuration::deleteByName(self::$URL_STATIC_OC_CK);
                Configuration::deleteByName(self::$URL_API_EMPATHYBROKER);
                Configuration::deleteByName(self::$SCRIPT_LOAD_EVENTS);
                Configuration::deleteByName(self::$SCRIPT_LOAD_TIMER);
            } elseif ((bool) Tools::isSubmit('oct_search_engine_changed') == true) {
                $engine = pSQL(trim(strip_tags(Tools::getValue(self::$SEARCH_ENGINE))));
                Configuration::updateValue(self::$SEARCH_ENGINE, $engine);

                $id_shop = (int) $this->getContext()->shop->id;
                $languages = Language::getLanguages(true, $id_shop);

                foreach ($languages as $language) {
                    // $iso_code = strtoupper($language['iso_code']);
                    $iso_code = $language['language_code'];

                    $getInputValue = function ($configKey) use ($iso_code) {
                        return pSQL(trim(strip_tags(Tools::getValue($configKey . '_' . $iso_code))));
                    };

                    // Guardar valores para cada campo con sufijo de idioma
                    $url_type_empathybroker = $getInputValue(self::$URL_API_EMPATHYBROKER) ?: 'https://api.empathybroker.com/search/v1/query/INSTANCE_ID/search?q=';
                    $endpoint_motiveSearch = $getInputValue(self::$ENDPOINT_APISEARCH) ?: self::$DEFAULT_ENDPOINT_APISEARCH;
                    $indexId_motiveSearch = $getInputValue(self::$INDEXID_APISEARCH);
                    $site_motiveSearch = $getInputValue(self::$SITE_APISEARCH);
                    $url_type_motiveSearch = $getInputValue(self::$URL_API_MOTIVESEARCH) ?: 'https://search.api.motive.co/search';
                    $tenant_type_motiveSearch = $getInputValue(self::$TENANT_MOTIVESEARCH);
                    $doofinderHashId = $getInputValue(self::$DOOFINDER_HASH_ID);
                    $doofinderApiKey = $getInputValue(self::$DOOFINDER_API_KEY);
                    $doofinderSearchZone = $getInputValue(self::$DOOFINDER_SEARCH_ZONE);
                    $doofinderQueryName = $getInputValue(self::$DOOFINDER_QUERY_NAME);
                    $doofinderIndices = $getInputValue(self::$DOOFINDER_INDICES);
                    $doofinderProductIdProperty = $getInputValue(self::$DOOFINDER_PRODUCTID_PROPERTY);

                    Configuration::updateValue(self::$URL_API_EMPATHYBROKER . '_' . $iso_code, $url_type_empathybroker);
                    Configuration::updateValue(self::$ENDPOINT_APISEARCH . '_' . $iso_code, $endpoint_motiveSearch);
                    Configuration::updateValue(self::$INDEXID_APISEARCH . '_' . $iso_code, $indexId_motiveSearch);
                    Configuration::updateValue(self::$SITE_APISEARCH . '_' . $iso_code, $site_motiveSearch);
                    Configuration::updateValue(self::$URL_API_MOTIVESEARCH . '_' . $iso_code, $url_type_motiveSearch);
                    Configuration::updateValue(self::$TENANT_MOTIVESEARCH . '_' . $iso_code, $tenant_type_motiveSearch);
                    Configuration::updateValue(self::$DOOFINDER_API_KEY . '_' . $iso_code, $doofinderApiKey);
                    Configuration::updateValue(self::$DOOFINDER_HASH_ID . '_' . $iso_code, $doofinderHashId);
                    Configuration::updateValue(self::$DOOFINDER_SEARCH_ZONE . '_' . $iso_code, $doofinderSearchZone);
                    Configuration::updateValue(self::$DOOFINDER_QUERY_NAME . '_' . $iso_code, $doofinderQueryName);
                    Configuration::updateValue(self::$DOOFINDER_INDICES . '_' . $iso_code, $doofinderIndices);
                    Configuration::updateValue(self::$DOOFINDER_PRODUCTID_PROPERTY . '_' . $iso_code, $doofinderProductIdProperty);
                }
            } elseif ((bool) Tools::isSubmit('oct_options_changed') == true) {
                $position = pSQL(trim(strip_tags(Tools::getValue(self::$POSITION_LOAD)))); // posicion donde se cargara el js de oct8ne
                $url_img_type = pSQL(trim(strip_tags(Tools::getValue(self::$URL_IMG_TYPE)))); // image url type
                $order_details_by = pSQL(trim(strip_tags(Tools::getValue(self::$ORDER_DETAILS_BY)))); // order details by
                $script_events = pSQL(trim(strip_tags(Tools::getValue(self::$SCRIPT_LOAD_EVENTS))));
                $script_timer = pSQL(trim(strip_tags(Tools::getValue(self::$SCRIPT_LOAD_TIMER))));

                $client_extra_script = trim(strip_tags(Tools::getValue(self::$CLIENT_EXTRA_SCRIPT)));

                if (isset($client_extra_script)) {
                    Configuration::updateValue(self::$CLIENT_EXTRA_SCRIPT, $client_extra_script);
                }

                // update values
                Configuration::updateValue(self::$POSITION_LOAD, $position);
                Configuration::updateValue(self::$URL_IMG_TYPE, $url_img_type);
                Configuration::updateValue(self::$ORDER_DETAILS_BY, $order_details_by);
                Configuration::updateValue(self::$SCRIPT_LOAD_EVENTS, $script_events);
                Configuration::updateValue(self::$SCRIPT_LOAD_TIMER, $script_timer);
            }
        } catch (Exception $ex) {
            $this->context->controller->errors[] = $ex->getMessage();
        }
    }

    /**
     * Muestra el formulario de inicio de sesion
     * inicio sesion
     *
     * @return mixed
     */
    private function renderSettingsForm()
    {
        // Esquleto del formulario
        $form_schema = [];
        // Configuración General
        $form_schema[] = $this->getSettingsFormSchema();

        $helper = new HelperForm();

        $helper->show_toolbar = false;
        $helper->table = $this->table;
        $helper->module = $this;

        $helper->identifier = $this->identifier;
        $helper->submit_action = 'login';
        $helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false) . '&configure=' . $this->name . '&tab_module=' . $this->tab . '&module_name=' . $this->name;
        $helper->token = Tools::getAdminTokenLite('AdminModules');

        $helper->tpl_vars = [
            'fields_value' => [self::$EMAIL_CK => pSQL(Tools::getValue(self::$EMAIL_CK)), self::$PASSWORD_CK => pSQL(Tools::getValue(self::$PASSWORD_CK))], /* Add values for your inputs */
        ];

        return $helper->generateForm($form_schema);
    }

    /**
     * Esquema del formulario de inicio de sesion
     *
     * @return array
     */
    private function getSettingsFormSchema()
    {
        $id_lang = (int) $this->context->employee->id_lang;
        $lang = new Language($id_lang);
        $iso = $lang->locale;
        $safe_iso = htmlspecialchars($iso, ENT_QUOTES, 'UTF-8');

        return [
            'form' => [
                'legend' => [
                    'title' => $this->l('Oct8ne Settings'),
                    'icon' => 'icon-cogs',
                ],
                'input' => [
                    [
                        'type' => 'text',
                        'name' => self::$EMAIL_CK,
                        'required' => true,
                        'label' => $this->l('Email'),
                        'class' => 'col-lg-4',
                        'placeholder' => $this->l('Your email...'),
                        'prefix' => "<i class='icon-envelope'></i>",
                    ],
                    [
                        'type' => 'password',
                        'name' => self::$PASSWORD_CK,
                        'required' => true,
                        'label' => $this->l('Password'),
                        'placeholder' => $this->l('Your password'),
                        'class' => 'col-lg-6',
                    ],
                    [
                        'type' => 'html',
                        'name' => 'help',
                        'html_content' => $this->l('You must fill out the fields with your user information from the admin panel on Oct8ne. If you still do not have an Oct8ne user name, you can create one') . " <a href='https://secure.oct8ne.com/SignUp/StepOne?lang={$safe_iso}' target='_blank' >" . $this->l('here') . '</a>',
                    ],
                ],
                'submit' => [
                    'title' => $this->l('Login'),
                ],
            ],
        ];
    }

    /**
     * Muestra el formulario de cerrar sesion
     *
     * @return mixed
     */
    private function renderLoggedForm()
    {
        // Esquleto del formulario
        $form_schema = [];
        // Configuración General
        $form_schema[] = $this->getLoggedFormSchema();

        $helper = new HelperForm();

        $helper->show_toolbar = false;
        $helper->table = $this->table;
        $helper->module = $this;

        $helper->identifier = $this->identifier;
        $helper->submit_action = 'logout';
        $helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false) . '&configure=' . $this->name . '&tab_module=' . $this->tab . '&module_name=' . $this->name;
        $helper->token = Tools::getAdminTokenLite('AdminModules');

        $helper->tpl_vars = [
            'fields_value' => ['email' => Configuration::get(self::$EMAIL_CK)], /* Add values for your inputs */
        ];

        return $helper->generateForm($form_schema);
    }

    /**
     * Muestra el formulario de metodo de busqueda
     *
     * @return mixed
     */
    private function renderSearchEngineForm()
    {
        // Obtenemos los motores de búsqueda
        $search_engines = $this->getDetectedSearchEngines();

        // Esquema del formulario
        $form_schema = [];
        // Configuración General
        $form_schema[] = $this->getSearchEngineFormSchema($search_engines);

        $helper = new HelperForm();

        $helper->show_toolbar = false;
        $helper->table = $this->table;
        $helper->module = $this;

        $helper->identifier = $this->identifier;
        $helper->submit_action = 'oct_search_engine_changed';
        $helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false) . '&configure=' . $this->name . '&tab_module=' . $this->tab . '&module_name=' . $this->name;
        $helper->token = Tools::getAdminTokenLite('AdminModules');

        $id_shop = (int) $this->getContext()->shop->id;
        $languages = Language::getLanguages(true, $id_shop);

        $fields_values = [
            self::$SEARCH_ENGINE => Configuration::get(self::$SEARCH_ENGINE),
        ];

        foreach ($languages as $language) {
            // $iso_code = strtoupper($language['iso_code']);
            $iso_code = $language['language_code'];

            $getConfigValue = function ($configKey) use ($iso_code) {
                $configValue = Configuration::get($configKey . '_' . $iso_code);

                if (empty($configValue)) {
                    $configValue = Configuration::get($configKey);
                }

                return $configValue;
            };

            $fields_values[self::$URL_API_EMPATHYBROKER . '_' . $iso_code] = $getConfigValue(self::$URL_API_EMPATHYBROKER);
            $fields_values[self::$ENDPOINT_APISEARCH . '_' . $iso_code] = $getConfigValue(self::$ENDPOINT_APISEARCH) ?: self::$DEFAULT_ENDPOINT_APISEARCH;
            $fields_values[self::$INDEXID_APISEARCH . '_' . $iso_code] = $getConfigValue(self::$INDEXID_APISEARCH);
            $fields_values[self::$SITE_APISEARCH . '_' . $iso_code] = $getConfigValue(self::$SITE_APISEARCH);
            $fields_values[self::$URL_API_MOTIVESEARCH . '_' . $iso_code] = $getConfigValue(self::$URL_API_MOTIVESEARCH);
            $fields_values[self::$TENANT_MOTIVESEARCH . '_' . $iso_code] = $getConfigValue(self::$TENANT_MOTIVESEARCH);
            $fields_values[self::$DOOFINDER_API_KEY . '_' . $iso_code] = $getConfigValue(self::$DOOFINDER_API_KEY);
            $fields_values[self::$DOOFINDER_HASH_ID . '_' . $iso_code] = $getConfigValue(self::$DOOFINDER_HASH_ID);
            $fields_values[self::$DOOFINDER_SEARCH_ZONE . '_' . $iso_code] = $getConfigValue(self::$DOOFINDER_SEARCH_ZONE);
            $fields_values[self::$DOOFINDER_QUERY_NAME . '_' . $iso_code] = $getConfigValue(self::$DOOFINDER_QUERY_NAME);
            $fields_values[self::$DOOFINDER_INDICES . '_' . $iso_code] = $getConfigValue(self::$DOOFINDER_INDICES);
            $fields_values[self::$DOOFINDER_PRODUCTID_PROPERTY . '_' . $iso_code] = $getConfigValue(self::$DOOFINDER_PRODUCTID_PROPERTY);
        }

        $helper->tpl_vars = [
            'fields_value' => $fields_values,
        ];

        return $helper->generateForm($form_schema);
    }

    /**
     * Devuelve una lista de motores de búsqueda detectados
     *
     * @return array
     */
    public function getDetectedSearchEngines()
    {
        // Cargamos la libreria para determina los tipos soportados
        $this->loadLibrary('OctSearchEngineType', 'search');
        $search_engines = [];

        // Por defecto
        $search_engines[OctSearchEngineType::Internal] = $this->l('Default prestashop search engine');
        $search_engines[OctSearchEngineType::Empathybroker] = $this->l('Empathybroker search engine');
        $search_engines[OctSearchEngineType::MotiveSearch] = $this->l('Motive search engine');
        $search_engines[OctSearchEngineType::ApiSearch] = $this->l('ApiSearch engine');
        $search_engines[OctSearchEngineType::EcommFinderSearch] = $this->l('EcommFinder Search engine');
        $search_engines[OctSearchEngineType::Doofinder] = $this->l('Doofinder search engine');

        if (Module::isEnabled('ambjolisearch')) {
            $search_engines[OctSearchEngineType::JolieSearch] = $this->l('JoliSearch');
        }

        return $search_engines;
    }

    /**
     * Esquema del formulario para cambiar el metodo de busqueda
     *
     * @return array
     */
    private function getSearchEngineFormSchema($search_engines)
    {
        $options = [];
        // Añadimos cada motor detectado
        foreach ($search_engines as $id => $name) {
            $options[] = [
                'id_option' => $id,
                'name' => $name,
            ];
        }

        $optionsQueryName = [
            [
                'id_option' => 'none',
                'name' => 'none',
            ],
            [
                'id_option' => 'match_and',
                'name' => 'match_and',
            ],
            [
                'id_option' => 'match_or',
                'name' => 'match_or',
            ],
            [
                'id_option' => 'fuzzy',
                'name' => 'fuzzy',
            ],
        ];

        $searchEngineSchema = [
            'form' => [
                'legend' => [
                    'title' => $this->l('Oct8ne search engine'),
                    'icon' => 'icon-cogs',
                ],
                'input' => [
                    [
                        'type' => 'select',
                        'required' => true,
                        'label' => $this->l('Search engine'),
                        'name' => self::$SEARCH_ENGINE,
                        'class' => 'search_engine',
                        'desc' => ' ',
                        'options' => [
                            'query' => $options,
                            'id' => 'id_option',
                            'name' => 'name',
                        ],
                    ],
                ],
                'submit' => [
                    'title' => $this->l('Save changes'),
                    'icon' => 'process-icon-save',
                ],
            ],
        ];

        $id_shop = (int) $this->getContext()->shop->id;
        $languages = Language::getLanguages(true, $id_shop);

        foreach ($languages as $language) {
            $languageCode = $language['language_code'];
            $languageName = $language['name'];

            $this->context->smarty->assign('languageName', $languageName);
            $separatorHtml = $this->context->smarty->fetch(_PS_MODULE_DIR_ . 'oct8ne/views/templates/admin/_partials/language_separator.tpl');

            $searchEngineSchema['form']['input'][] = [
                'type' => 'html',
                'name' => 'separator_' . $languageCode,
                'html_content' => $separatorHtml,
            ];

            $searchEngineSchema['form']['input'][] = [
                'type' => 'text',
                'label' => $this->l('URL API') . ' (' . $languageCode . ')',
                'name' => self::$URL_API_EMPATHYBROKER . '_' . $languageCode,
                'class' => 'url_api_empathybroker',
                'desc' => ' ',
                'size' => 75,
            ];
            $searchEngineSchema['form']['input'][] = [
                'type' => 'text',
                'label' => $this->l('URL MOTIVE API') . ' (' . $languageCode . ')',
                'name' => self::$URL_API_MOTIVESEARCH . '_' . $languageCode,
                'class' => 'url_api_motiveSearch',
                'desc' => ' ',
                'size' => 75,
            ];
            $searchEngineSchema['form']['input'][] = [
                'type' => 'text',
                'label' => $this->l('TENANT MOTIVESEARCH') . ' (' . $languageCode . ')',
                'name' => self::$TENANT_MOTIVESEARCH . '_' . $languageCode,
                'class' => 'tenant_motiveSearch',
                'desc' => ' ',
                'size' => 75,
            ];
            $searchEngineSchema['form']['input'][] = [
                'type' => 'text',
                'label' => $this->l('EndPoint') . ' (' . $languageCode . ')',
                'name' => self::$ENDPOINT_APISEARCH . '_' . $languageCode,
                'class' => 'endpoint_apiSearch',
                'desc' => ' ',
                'size' => 75,
            ];
            $searchEngineSchema['form']['input'][] = [
                'type' => 'text',
                'label' => $this->l('Index ID') . ' (' . $languageCode . ')',
                'name' => self::$INDEXID_APISEARCH . '_' . $languageCode,
                'class' => 'indexid_apiSearch',
                'desc' => ' ',
                'size' => 75,
            ];
            $searchEngineSchema['form']['input'][] = [
                'type' => 'text',
                'label' => $this->l('Site') . ' (' . $languageCode . ')',
                'name' => self::$SITE_APISEARCH . '_' . $languageCode,
                'class' => 'site_apiSearch',
                'desc' => ' ',
                'size' => 75,
            ];
            $searchEngineSchema['form']['input'][] = [
                'type' => 'text',
                'required' => true,
                'label' => $this->l('DOOFINDER SEARCH ZONE') . ' (' . $languageCode . ')',
                'name' => self::$DOOFINDER_SEARCH_ZONE . '_' . $languageCode,
                'class' => 'doofinder_search_zone',
                'desc' => ' ',
                'size' => 75,
            ];
            $searchEngineSchema['form']['input'][] = [
                'type' => 'text',
                'required' => true,
                'label' => $this->l('DOOFINDER API KEY') . ' (' . $languageCode . ')',
                'name' => self::$DOOFINDER_API_KEY . '_' . $languageCode,
                'class' => 'doofinder_api_key',
                'desc' => ' ',
                'size' => 75,
            ];
            $searchEngineSchema['form']['input'][] = [
                'type' => 'text',
                'required' => true,
                'label' => $this->l('DOOFINDER HASH ID') . ' (' . $languageCode . ')',
                'name' => self::$DOOFINDER_HASH_ID . '_' . $languageCode,
                'class' => 'doofinder_hash_id',
                'desc' => ' ',
                'size' => 75,
            ];
            $searchEngineSchema['form']['input'][] = [
                'type' => 'text',
                'label' => $this->l('DOOFINDER INDICES') . ' (' . $languageCode . ')',
                'name' => self::$DOOFINDER_INDICES . '_' . $languageCode,
                'class' => 'doofinder_indices',
                'desc' => 'Your search engine is composed by one or many Indices. With this config you can specify to search within one specific Index. If this parameter is not provided, the search will work with all Indices. You can specify multiple index separated by comma. These indices must all exist',
                'size' => 75,
            ];
            $searchEngineSchema['form']['input'][] = [
                'type' => 'text',
                'label' => $this->l('DOOFINDER PRODUCTID PROPERTY') . ' (' . $languageCode . ')',
                'name' => self::$DOOFINDER_PRODUCTID_PROPERTY . '_' . $languageCode,
                'class' => 'doofinder_productId_property',
                'desc' => 'The name of the property on doofinder feed that contains the product ID  from prestashop database. If not provided, will use "id" property by default.',
                'size' => 75,
            ];
            $searchEngineSchema['form']['input'][] = [
                'type' => 'select',
                'required' => true,
                'label' => $this->l('DOOFINDER QUERY TYPE') . ' (' . $languageCode . ')',
                'name' => self::$DOOFINDER_QUERY_NAME . '_' . $languageCode,
                'class' => 'doofinder_query_name',
                'desc' => 'none ==> Doofinder search service uses the best query for the search being made.</br>match_and ==> Doofinder will return results that contain all the search terms.</br>match_or ==> Doofinder will return results that contain any of the search terms. Of course, results that contain all the terms are scored better.</br>fuzzy ==> Doofinder will try to apply some «fuzzy logic» to determine whether a result, even if it’s not an exact match, is «close enough» to the search terms.',
                'options' => [
                    'query' => $optionsQueryName,
                    'id' => 'id_option',
                    'name' => 'name',
                ],
            ];
        }

        return $searchEngineSchema;
    }

    /**
     * Esquema del formulario cerrar sesion
     *
     * @return array
     */
    private function getLoggedFormSchema()
    {
        $this->context->smarty->assign('oct_mail', Configuration::get(self::$EMAIL_CK));

        return [
            'form' => [
                'legend' => [
                    'title' => $this->l('Oct8ne info'),
                    'icon' => 'icon-cogs',
                ],
                'input' => [
                    [
                        'type' => 'html',
                        'required' => true,
                        'label' => $this->l('Active User: '),
                        'name' => 'email',
                        'html_content' => $this->display(__FILE__, 'views/templates/admin/mail.tpl'),
                    ],
                ],

                'submit' => [
                    'title' => $this->l('Logout'),
                    'icon' => 'process-icon-close',
                ],
            ],
        ];
    }

    /**
     * Muestra el formulario para cambiar la posicion donde se carga el codigo de Oct8ne
     *
     * @return mixed
     */
    public function renderPositionLoadForm()
    {        // Esquleto del formulario
        $form_schema = [];
        // Configuración General
        $form_schema[] = $this->getPositionLoadFormSchema();

        $helper = new HelperForm();

        $helper->show_toolbar = false;
        $helper->table = $this->table;
        $helper->module = $this;

        $helper->identifier = $this->identifier;
        $helper->submit_action = 'oct_options_changed';
        $helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false) . '&configure=' . $this->name . '&tab_module=' . $this->tab . '&module_name=' . $this->name;
        $helper->token = Tools::getAdminTokenLite('AdminModules');

        $helper->tpl_vars = [
            'fields_value' => [self::$POSITION_LOAD => Configuration::get(self::$POSITION_LOAD), self::$URL_IMG_TYPE => Configuration::get(self::$URL_IMG_TYPE), self::$ORDER_DETAILS_BY => Configuration::get(self::$ORDER_DETAILS_BY), self::$CLIENT_EXTRA_SCRIPT => Configuration::get(self::$CLIENT_EXTRA_SCRIPT), self::$SCRIPT_LOAD_EVENTS => Configuration::get(self::$SCRIPT_LOAD_EVENTS), self::$SCRIPT_LOAD_TIMER => Configuration::get(self::$SCRIPT_LOAD_TIMER)], /* Add values for your inputs */
        ];

        return $helper->generateForm($form_schema);
    }

    /**
     * Esquema del formulario para cambiar la posicion donde se carga el codigo de Oct8ne
     *
     * @return array
     */
    private function getPositionLoadFormSchema()
    {
        $this->context->smarty->assign(self::$POSITION_LOAD, Configuration::get(self::$POSITION_LOAD));
        $this->context->smarty->assign(self::$URL_IMG_TYPE, Configuration::get(self::$URL_IMG_TYPE));
        $this->context->smarty->assign(self::$CLIENT_EXTRA_SCRIPT, Configuration::get(self::$CLIENT_EXTRA_SCRIPT));
        $this->context->smarty->assign(self::$SCRIPT_LOAD_EVENTS, Configuration::get(self::$SCRIPT_LOAD_EVENTS));
        $this->context->smarty->assign(self::$SCRIPT_LOAD_TIMER, Configuration::get(self::$SCRIPT_LOAD_TIMER));

        $options = [];
        // Añadimos cada motor detectado
        $options[] = [
            'id_option' => 1,
            'name' => $this->l('On Header'),
        ];

        $options[] = [
            'id_option' => 2,
            'name' => $this->l('On Footer'),
        ];

        $options2 = [];
        // Añadimos cada motor detectado
        $options2[] = [
            'id_option' => 1,
            'name' => $this->l('Standard'),
        ];

        $options2[] = [
            'id_option' => 2,
            'name' => 'Type 1',
        ];

        $options3 = [];
        $options3[] = [
            'id_option' => 'DISABLED',
            'name' => $this->l('Disabled'),
        ];
        $options3[] = [
            'id_option' => 'ALL',
            'name' => 'Any user event (Click, Scroll, Mousemove)',
        ];
        $options3[] = [
            'id_option' => 'scroll',
            'name' => 'Scroll',
        ];
        $options3[] = [
            'id_option' => 'click',
            'name' => 'Click',
        ];
        $options3[] = [
            'id_option' => 'mousemove',
            'name' => 'Mousemove',
        ];
        $options3[] = [
            'id_option' => 'SCRIPT',
            'name' => 'Script call',
        ];

        $options4 = [];
        $options4[] = [
            'id_option' => 0,
            'name' => $this->l('Disabled'),
        ];
        $options4[] = [
            'id_option' => 1,
            'name' => '1 sec',
        ];
        $options4[] = [
            'id_option' => 2,
            'name' => '2 secs',
        ];
        $options4[] = [
            'id_option' => 3,
            'name' => '3 secs',
        ];
        $options4[] = [
            'id_option' => 4,
            'name' => '4 secs',
        ];
        $options4[] = [
            'id_option' => 5,
            'name' => '5 secs',
        ];
        $options4[] = [
            'id_option' => 6,
            'name' => '6 secs',
        ];
        $options4[] = [
            'id_option' => 7,
            'name' => '7 secs',
        ];
        $options4[] = [
            'id_option' => 8,
            'name' => '8 secs',
        ];
        $options4[] = [
            'id_option' => 9,
            'name' => '9 secs',
        ];
        $options4[] = [
            'id_option' => 10,
            'name' => '10 secs',
        ];

        return [
            'form' => [
                'legend' => [
                    'title' => $this->l('Oct8ne options'),
                    'icon' => 'icon-cogs',
                ],

                'input' => [
                    [
                        'type' => 'select',
                        'required' => true,
                        'desc' => $this->l('You can choose the position to load Oct8ne scripts (On Footer or On Header) of the page'),
                        'label' => $this->l('Position'),
                        'name' => self::$POSITION_LOAD,
                        'options' => [
                            'query' => $options,
                            'id' => 'id_option',
                            'name' => 'name',
                        ],
                    ],
                    [
                        'type' => 'select',
                        'required' => true,
                        'label' => $this->l('Url image type'),
                        'desc' => $this->l('Type 1 adds product id before image id. Useful in specific instances. Be careful, change this option only when necessary'),
                        'name' => self::$URL_IMG_TYPE,
                        'options' => [
                            'query' => $options2,
                            'id' => 'id_option',
                            'name' => 'name',
                        ],
                    ],
                    [
                        'type' => 'text',
                        'required' => false,
                        'label' => $this->l('Get order details by'),
                        'desc' => $this->l('Select the parameter to get the order details by in the query'),
                        'name' => self::$ORDER_DETAILS_BY,
                    ],
                    [
                        'type' => 'select',
                        'required' => true,
                        'desc' => $this->l('**If you choose "Script call" you must call the function insertOct8ne();whenever you want load Oct8ne\'s code on the page.**'),
                        'label' => $this->l('Delay Oct8ne loading until user interaction:'),
                        'name' => self::$SCRIPT_LOAD_EVENTS,
                        'options' => [
                            'query' => $options3,
                            'id' => 'id_option',
                            'name' => 'name',
                        ],
                    ],
                    [
                        'type' => 'select',
                        'required' => true,
                        'desc' => $this->l(''),
                        'label' => $this->l('Delay Oct8ne loading on page (seconds):'),
                        'name' => self::$SCRIPT_LOAD_TIMER,
                        'options' => [
                            'query' => $options4,
                            'id' => 'id_option',
                            'name' => 'name',
                        ],
                    ],
                ],
                'submit' => [
                    'title' => $this->l('Save'),
                    'icon' => 'process-icon-save',
                ],
            ],
        ];
    }

    /**
     * Conecta con la API REST de Oct8ne y comprueba los datos de usuario
     *
     * @param $user
     * @param $pass
     *
     * @return array|bool
     */
    private function checkOct8neLinkUp($user, $pass)
    {
        try {
            // peticion
            $url = 'https://backoffice.oct8ne.com/platformConnection/linkup';
            $data = [
                'email' => $user,
                'pass' => $pass,
                'platform' => 'prestashop',
                'urlDomain' => $this->getContext()->shop->domain_ssl,
                'statusPlatform' => $this->active == 1,
            ];

            $encodedData = '';
            $encodedData = json_encode($data);

            $options = [
                'http' => [
                    'header' => 'Content-Type: application/json;charset=UTF-8',
                    'method' => 'POST',
                    'content' => $encodedData,
                ],
            ];
            $context = stream_context_create($options);
            $result = self::OctFileGetContents($url, false, $context);
            $result = json_decode($result);

            if (isset($result)) {
                // si se devuelve una licencia y token correctos se devuelve
                $license = $result->LicenseId;
                $token = $result->ApiToken;
                $server = $result->Server;
                $UrlStatic = $result->UrlStatic;
                // $msg = $result->Message;

                if ($license != null && $token != null) {
                    return ['license' => $license, 'token' => $token, 'server' => $server, 'urlstatic' => $UrlStatic];
                } else {
                    return false;
                }
            } else {
                return false;
            }
        } catch (Exception $ex) {
            return false;
        }
    }

    private static function octFileGetContents($url, $use_include_path = false, $stream_context = null, $curl_timeout = 5)
    {
        if ($stream_context == null && preg_match('/^https?:\/\//', $url)) {
            $stream_context = @stream_context_create([
                'http' => ['timeout' => $curl_timeout],
                'ssl' => [
                    'verify_peer' => true,
                    'verify_peer_name' => true,
                    'disable_compression' => true,
                    'allow_self_signed' => false,
                    'ciphers' => 'HIGH:!SSLv2:!SSLv3',
                ],
            ]);
        }

        $return = false;

        if (in_array(ini_get('allow_url_fopen'), ['On', 'on', '1']) || !preg_match('/^https?:\/\//', $url)) {
            $meth = 'file_get_contents';
            $return = @$meth($url, $use_include_path, $stream_context);
        }

        if ($return === false && function_exists('curl_init')) {
            $curl = curl_init();
            curl_setopt($curl, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($curl, CURLOPT_URL, $url);
            curl_setopt($curl, CURLOPT_CONNECTTIMEOUT, 5);
            curl_setopt($curl, CURLOPT_TIMEOUT, $curl_timeout);

            // SOLUCIÓN AQUÍ: configurar SSL correctamente
            // curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, true);
            // curl_setopt($curl, CURLOPT_SSL_VERIFYHOST, 2);
            curl_setopt($curl, CURLOPT_SSL_CIPHER_LIST, 'HIGH:!SSLv2:!SSLv3');

            curl_setopt($curl, CURLOPT_HTTPHEADER, ['Content-Type: application/json;charset=UTF-8']);

            if ($stream_context != null) {
                $opts = stream_context_get_options($stream_context);

                if (isset($opts['http']['method']) && Tools::strtolower($opts['http']['method']) == 'post') {
                    curl_setopt($curl, CURLOPT_POST, true);

                    if (isset($opts['http']['content'])) {
                        curl_setopt($curl, CURLOPT_POSTFIELDS, $opts['http']['content']);
                    }
                }
            }
            $content = curl_exec($curl);
            curl_close($curl);
            $return = $content;
        }

        return $return;
    }

    private function formatPrice($price, $currency = null)
    {
        if (!$currency) {
            $currency = $this->context->currency;
        }

        if (version_compare(_PS_VERSION_, '1.7.7.0', '>=')) {
            // Desde PS 1.7.6 usamos el formateador de Locale

            $locale = $this->context->getCurrentLocale();

            return $locale->formatPrice($price, $currency->iso_code);
        } else {
            // Compatibilidad con PS 1.5 a 1.7.5
            return $this->legacyDisplayPrice($price, $currency);
        }
    }

    private function legacyDisplayPrice($price, $currency)
    {
        // Solo PS < 1.7.6 usará esto
        $method = 'displayPrice';

        return Tools::$method($price, $currency);
    }

    /*
     * Hook actionValidateOrder
     * Es llamado al ser creado un pedido y crear el pixel de confirmacion de venta.
     */
    public function hookDisplayOrderConfirmation($params)
    {
        $license = Configuration::get(self::$LICENSE_ID_CK);
        $server = Configuration::get(self::$SERVER_OC_CK);
        $urlstatic = Configuration::get(self::$URL_STATIC_OC_CK);

        if (!empty($license) && !empty($server) && !empty($urlstatic)) {
            $order = version_compare(_PS_VERSION_, '1.7', '<') ? $params['objOrder'] : $params['order'];

            if ($order) {
                $currency = Currency::getCurrency($order->id_currency);
                $isoLang = Language::getIsoById($order->id_lang);
                $total = $this->formatPrice($order->total_paid);

                if (array_key_exists('sign', $currency)) {
                    $total = str_replace($currency['sign'], '', $total);
                }
                $total = str_replace(' ', '', $total);

                $this->context->smarty->assign([
                    'license' => $license,
                    'urlstatic' => $urlstatic,
                    'currency_code' => $currency['iso_code'],
                    'locale' => $isoLang . '-' . Tools::strtoupper($isoLang),
                    'value' => $total,
                    'reference' => $order->reference,
                ]);

                return $this->context->smarty->fetch(_PS_MODULE_DIR_ . 'oct8ne/views/templates/hook/sale_notification.tpl');
            }
        }
    }

    /**
     * Hook footer
     */
    public function hookDisplayFooter($params)
    {
        $position = Configuration::get(self::$POSITION_LOAD);

        if ($position == 2) {
            return $this->octDisplayCode();
        }

        return '';
    }

    /**
     * Hook header
     */
    public function hookDisplayHeader($params)
    {
        $position = Configuration::get(self::$POSITION_LOAD);

        if ($position == 1) {
            return $this->octDisplayCode();
        }

        return '';
    }

    /**
     * Codigo JS para el footer, solo disponible si existe licencia de oct8ne
     *
     * @param $params
     *
     * @return bool
     */
    public function octDisplayCode()
    {
        $license = Configuration::get(self::$LICENSE_ID_CK);

        if (!empty($license)) {
            $baseUrl = rtrim($this->context->shop->getBaseURL(), '/');

            $loginUrl = $this->context->link->getPageLink('authentication', true);

            $checkoutSuccessUrl = '';
            $aux = Configuration::get('PS_ORDER_PROCESS_TYPE');

            if ($aux == 0) {
                $checkoutSuccessUrl = $this->context->link->getPageLink('order', true);
            } elseif ($aux == 1) {
                $checkoutSuccessUrl = $this->context->link->getPageLink('order-opc', true);
            }

            $orderConfirmationUrl = $this->context->link->getPageLink('order-confirmation', true);

            $locale = $this->context->language->language_code;

            if (strpos($locale, '-') == true) {
                $aux = explode('-', $locale);
                $aux[1] = Tools::strtoupper($aux[1]);
                $locale = implode('-', $aux);
            }
            $currencyCode = $this->context->currency->iso_code;

            // /Comprobar si estamos es una pagina de producto
            $controller = $this->context->controller;

            if ($this->isaProductController()) {
                $product = new Product(pSQL(Tools::getValue('id_product')), false, $this->getContext()->language->id);

                if (Validate::isLoadedObject($product) && ($product->price > 0 || $product->getPrice() > 0)) {
                    $id = $product->id;
                    $image = Product::getCover($id);
                    $image = $image['id_image'];

                    $thumbnail = $this->getContext()->link->getImageLink($product->link_rewrite, $image);

                    $this->context->smarty->assign([
                        'oct8ne_product_id' => $id,
                        'oct8ne_product_thumb' => $thumbnail,
                    ]);
                }
            }

            $baseUrl = Oct8ne::removeHttProtocol($baseUrl);
            $loginUrl = Oct8ne::removeHttProtocol($loginUrl);
            $checkoutSuccessUrl = Oct8ne::removeHttProtocol($checkoutSuccessUrl);
            $orderConfirmationUrl = Oct8ne::removeHttProtocol($orderConfirmationUrl);

            $server = Configuration::get(self::$SERVER_OC_CK);
            $urlstatic = Configuration::get(self::$URL_STATIC_OC_CK);
            $clientextraData = Configuration::get(self::$CLIENT_EXTRA_SCRIPT);

            if (!empty($clientextraData)) {
                $this->context->smarty->assign('oct8neExtraData', $clientextraData);
            }

            $scriptTimer = Configuration::get(self::$SCRIPT_LOAD_TIMER);
            $scriptTimer = (int) $scriptTimer;

            $scriptEvents = Configuration::get(self::$SCRIPT_LOAD_EVENTS);

            $userLastName = null;
            $userFirstName = null;
            $userEmail = null;

            if (empty($server)) {
                $server = 'backoffice.oct8ne.com/';
            }

            if (empty($urlstatic)) {
                $urlstatic = 'static.oct8ne.com/';
            }

            $customer = $this->getContext()->customer;

            if (!empty($customer) && Validate::isLoadedObject($customer) && $customer->isLogged() && !$customer->deleted) {
                $userFirstName = $customer->firstname;

                if (!empty($customer->lastname)) {
                    $userLastName = $customer->lastname;
                }
                $userEmail = $customer->email;
            }

            $this->context->smarty->assign([
                'oct8neLicense' => $license,
                'oct8neBaseUrl' => $baseUrl,
                'oct8neLoginUrl' => $loginUrl,
                'oct8neCheckOutSuccessUrl' => $checkoutSuccessUrl,
                'oct8neLocale' => $locale,
                'oct8neCurrencyCode' => $currencyCode,
                'oct8neorderConfirmationUrl' => $orderConfirmationUrl,
                'oct8neServer' => $server,
                'oct8neUrlStatic' => $urlstatic,
                'oct8neScriptTimer' => $scriptTimer,
                'oct8neScriptEvents' => $scriptEvents,
                'oct8neUserFirstName' => $userFirstName,
                'oct8neUserLastName' => $userLastName,
                'oct8neUserEmail' => $userEmail,
            ]);

            return $this->display(__FILE__, 'hookDisplayOctCode.tpl');
        } else {
            return false;
        }
    }

    private function isaProductController()
    {
        // Método 1
        $controller = $this->context->controller;

        if (isset($controller->php_self)) {
            if ($controller->php_self == 'product') {
                return true;
            }
        }

        // Método 2 - Legacy
        $class = get_class($controller);

        return Tools::strtolower($class) == 'productcontroller';
    }

    /**
     * Regla que permite a Oct8ne Conectar con nuestro controlador
     */
    public function setHtaccessRules()
    {
        return true;
    }

    /**
     * Eliminar regla Oct8ne
     */
    public function removeHtaccessRules()
    {
    }
}
