<?php
/**
 * Prestashop module for Oct8ne
 *
 * @author    Prestaquality.com
 * @copyright 2016 Prestaquality
 * @license   Commercial license see license.txt *
 *
 * @category  Prestashop
 * @category  Module
 * Support by mail  : info@prestaquality.com
 */
if (!defined('_PS_VERSION_')) {
    exit;
}

require_once _PS_MODULE_DIR_ . 'oct8ne/lib/search/OctSearchFactory.php';

class Oct8neOct8neConnectorModuleFrontController extends ModuleFrontController
{
    /* public $ssl = true; */
    private $callback;

    private $octmethod;

    private $address;

    private $name_controller = 'oct8neconnector';

    private $include_subcategories = true;

    private function getContext()
    {
        return $this->context;
    }

    /**
     * Inicializacion
     */
    public function init()
    {
        parent::init();

        $this->callback = pSQL(trim(strip_tags(Tools::getValue('callback'))));
        $this->octmethod = pSQL(trim(strip_tags(Tools::getValue('octmethod'))));

        $this->address = null;
    }

    /**
     * Metodo que se ejecuta despues de la inicializacion. Comprueba si existe el metodo solicitado
     * y lo ejecuta si es el caso.
     */
    public function postProcess()
    {
        $data = '';
        try {
            $meth = $this->octmethod . 'OctMeth';

            if (method_exists($this, $meth)) {
                $data = $this->$meth();
            } else {
                throw new Exception('Method not Exists');
            }
        } catch (Exception $ex) {
            if (Tools::getIsset('debug')) {
                exit;
            }
        }
        $this->postProcessFinally($data);
    }

    /**
     * Acciones finales
     */
    private function postProcessFinally($data)
    {
        ob_end_clean();
        $this->printJsonCode($data);
        exit;
    }

    /**
     * Metodo que establece la moneda local
     */
    private function setContextCurrency()
    {
        $currency = pSQL(trim(strip_tags(Tools::getValue('currency'))));

        if (!empty($currency)) {
            $in_currency = Currency::getIdByIsoCode($currency);

            if (!empty($in_currency)) {
                // Cargamos el objeto, comprobamos que lo tenemos y que es válido
                $currency = new Currency($in_currency);

                if (Validate::isLoadedObject($currency) && $currency->active == '1') {
                    $this->getContext()->currency = $currency;
                }
            }
        }
    }

    /**
     * Metodo que establece la lengua local
     */
    private function setContextLanguage()
    {
        $language = pSQL(trim(strip_tags(Tools::getValue('locale'))));

        if (!empty($language)) {
            $global_local = explode('-', $language);

            $id_language = Language::getIdByIso($global_local[0]);

            if (!empty($id_language)) {
                // Cargamos el objeto, comprobamos que lo tenemos y que es valido
                $language = new Language($id_language);

                if (Validate::isLoadedObject($language) && $language->active == '1') {
                    $this->getContext()->language = $language;
                }
            }
        }
    }

    /**
     * Metodo que estable el id_country del pais se conectan
     */
    private function setContextCountry()
    {
        $this->address = Address::initialize();

        $iso_code_country = pSQL(trim(strip_tags(Tools::getValue('country'))));

        if (!empty($iso_code_country) && Validate::isGenericName($iso_code_country)) {
            $id_country = Country::getByIso($iso_code_country);

            if ($id_country) {
                $this->address->id_country = (int) $id_country;
            }
        }
    }

    private function getPrice($id_product, $usetax = true, $id_product_attribute = null, $usereduc = true)
    {
        $id_customer = null;
        $id_group = null;
        $specific_price_output = null;

        if (Validate::isLoadedObject($this->getContext()->customer)) {
            $id_customer = (int) $this->getContext()->customer->id;
            $id_group = (int) Customer::getDefaultGroupId($id_customer);
        } else {
            $id_group = (int) Group::getCurrent()->id;
        }

        $id_country = (int) $this->address->id_country;
        $id_state = (int) $this->address->id_state;
        $zipcode = $this->address->postcode;

        $return = Product::priceCalculation(
            $this->getContext()->shop->id,
            $id_product,
            $id_product_attribute,
            $id_country,
            $id_state,
            $zipcode,
            $this->getContext()->currency->id,
            $id_group,
            1,// quantity
            $usetax,
            6,// decimals
            false,// only_reduc
            $usereduc,// usereduc
            true,// with_ecotax
            $specific_price_output,// specific_price_output
            true,// use_group_reduction
            $id_customer,
            true,// use_customer_price
            0,// id_cart
            0,// cart_quantity
            0// id_customization
        );

        return $return;
    }

    /**
     * @param $data
     *              Imprime un json de respuesta
     */
    public function printJsonCode($data)
    {
        if (empty($this->callback)) {
            header('Content-Type: application/json; charset=UTF-8');
            $json = json_encode($data);

            echo $json;
        } else {
            $result = '';
            $result = json_encode($data);

            header('Content-Type: application/javascript; charset=UTF-8');
            $json = $this->callback . '(' . $result . ');';
            echo $json;
        }
    }

    private function formatPrice($price, $currency = null)
    {
        if (!$currency) {
            $currency = $this->context->currency;
        }

        if (version_compare(_PS_VERSION_, '1.7.7.0', '>=')) {
            // Desde PS 1.7.6 usamos el formateador de Locale

            $locale = $this->context->getCurrentLocale();

            return $locale->formatPrice($price, $currency->iso_code);
        } else {
            // Compatibilidad con PS 1.5 a 1.7.5
            return $this->legacyDisplayPrice($price, $currency);
        }
    }

    private function legacyDisplayPrice($price, $currency)
    {
        // Solo PS < 1.7 usará esto
        $method = 'displayPrice';

        return Tools::$method($price, $currency);
    }

    /**
     * Obtiene unos datos concretos sobre un producto, a modo de resumen simple
     *
     * @return array
     */
    public function getProductSummary($id_product)
    {
        $product = new Product($id_product, false, $this->getContext()->language->id);

        $result = [];

        if (Validate::isLoadedObject($product)) {
            if ($product->active) {
                $image = Product::getCover($id_product);
                $image = $image['id_image'];

                if (!isset($image)) {
                    $allImages = $product->getImages($this->getContext()->language->id);
                    foreach ($allImages as $firstImage) {
                        $image = $firstImage['id_image'];
                        break;
                    }
                }

                if (isset($image)) {
                    $thumbnail = $this->getContext()->link->getImageLink($product->link_rewrite, $image);
                    $url_image_type = Configuration::get('OCT_URL_IMG_TYPE');

                    if ($url_image_type == 2) {
                        $base_url = rtrim($this->context->shop->getBaseURL(), '/');
                        $base_url = str_replace('http://', '', $base_url);
                        $base_url = str_replace('https://', '', $base_url);
                        $base_url = $base_url . '/';

                        $thumbnail_exploded = explode($base_url, $thumbnail);
                        $thumbnail = $thumbnail_exploded[0] . $base_url . $id_product . '-' . $thumbnail_exploded[1];
                    }
                } else {
                    $thumbnail = '';
                }

                $allow_out_stock = true;

                if ($product->out_of_stock == 0) {
                    $allow_out_stock = false;
                } elseif ($product->out_of_stock == 2) {
                    $allow_out_stock = (bool) Configuration::get('PS_ORDER_OUT_OF_STOCK');
                }

                $result = [
                    'internalId' => $product->id,
                    'title' => $product->name,
                    'formattedPrice' => $this->formatPrice($this->getPrice($product->id, Product::$_taxCalculationMethod == PS_TAX_INC)),
                    'formattedPrevPrice' => $this->formatPrice($this->getPrice($product->id, Product::$_taxCalculationMethod == PS_TAX_INC, null, false)),
                    'productUrl' => $this->getContext()->link->getProductLink($id_product),
                    'thumbnail' => $thumbnail,
                    'configurable' => false,
                    'stock' => [
                        'quantity' => StockAvailable::getQuantityAvailableByProduct($product->id),
                        'minimal_quantity' => (int) $product->minimal_quantity,
                        'allowOutStock' => $allow_out_stock,
                        'labelInStock' => $product->available_now,
                        'labelOutStock' => $product->available_later,
                    ],
                ];

                $sqlAttributes = new DbQuery();
                $sqlAttributes->from('product_attribute', 'pa');
                $sqlAttributes->innerJoin('product_attribute_shop', 'pas', 'pa.id_product_attribute = pas.id_product_attribute');
                $sqlAttributes->where('pa.id_product = ' . (int) $product->id);
                $sqlAttributes->where('pas.id_shop = ' . (int) $this->context->shop->id);
                $sqlAttributes->orderBy('pa.`id_product_attribute`');
                $resultAttributes = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($sqlAttributes);

                if (Tools::strtolower($this->octmethod) == 'productinfo') {
                    if ($resultAttributes) {
                        $variations = [];
                        $attributes_group = [];

                        foreach ($resultAttributes as $i => $rowA) {
                            $id_product_attribute = (int) $rowA['id_product_attribute'];
                            $attributes = [];

                            $sqlAttributeGroup = new DbQuery();
                            $sqlAttributeGroup->select('agl.public_name, al.name');
                            $sqlAttributeGroup->from('product_attribute_combination', 'pac');
                            $sqlAttributeGroup->innerJoin('attribute', 'a', 'a.id_attribute = pac.id_attribute');
                            $sqlAttributeGroup->innerJoin('attribute_lang', 'al', 'al.id_attribute = a.id_attribute');
                            $sqlAttributeGroup->innerJoin('attribute_group', 'ag', 'ag.id_attribute_group = a.id_attribute_group');
                            $sqlAttributeGroup->innerJoin('attribute_group_lang', 'agl', 'agl.id_attribute_group = a.id_attribute_group');
                            $sqlAttributeGroup->where('pac.id_product_attribute = ' . (int) $id_product_attribute);
                            $sqlAttributeGroup->where('al.id_lang = ' . (int) $this->getContext()->language->id);
                            $sqlAttributeGroup->where('agl.id_lang = ' . (int) $this->getContext()->language->id);
                            $resultAttributeGroup = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($sqlAttributeGroup);

                            if ($resultAttributeGroup) {
                                foreach ($resultAttributeGroup as $rowG) {
                                    if ($i == 0) {
                                        $attributes_group[] = $rowG['public_name'];
                                    }

                                    $attributes[] = [
                                        $rowG['public_name'] => $rowG['name'],
                                    ];
                                }

                                $result['attributes'] = $attributes_group;
                                $result['variations'][] = [
                                    'id' => $id_product_attribute,
                                    'attributes' => $attributes,
                                    'formattedPrice' => $this->formatPrice($this->getPrice($product->id, Product::$_taxCalculationMethod == PS_TAX_INC, $id_product_attribute)),
                                    'formattedPrevPrice' => $this->formatPrice($this->getPrice($product->id, Product::$_taxCalculationMethod == PS_TAX_INC, $id_product_attribute, false)),
                                    'quantity' => (int) StockAvailable::getQuantityAvailableByProduct($product->id, $id_product_attribute),
                                    'minimal_quantity' => (int) $rowA['minimal_quantity'],
                                    'default' => (bool) $rowA['default_on'],
                                ];
                            }
                        }

                        unset($result['configurable']);
                    }
                } else {
                    $result['configurable'] = count($resultAttributes) > 0 ? true : false;
                }

                $samePrice = true;

                if (isset($result['variations'])) {
                    $provisionalPrice = null;
                    $variations = $result['variations'];

                    for ($i = 0; $i < count($variations); ++$i) {
                        if (isset($provisionalPrice)) {
                            if ($provisionalPrice != $variations[$i]['formattedPrice']) {
                                $samePrice = false;
                            }
                        } else {
                            $provisionalPrice = $variations[$i]['formattedPrice'];
                        }
                    }
                }

                // si tiene ipa los precios se devuelven como null
                $hasipa = ($product->cache_default_attribute > 0);

                if ($hasipa && !$samePrice) {
                    $result['formattedPrice'] = null;
                    $result['formattedPrevPrice'] = null;
                }
            }
        }

        return $result;
    }

    /**
     * Contiene informacion sobre el producto
     *
     * @return array
     */
    public function productInfoOctMeth()
    {
        $this->setContextLanguage();
        $this->setContextCurrency();
        // $this->setContextCustomer();
        $this->setContextCountry();
        $extraProperties = pSQL(trim(strip_tags(Tools::getValue('extraproperties'))));

        $ids = trim(strip_tags(Tools::getValue('productIds')));
        $ids = str_replace(' ', '', $ids);
        $ids = explode(',', $ids);
        $ids = array_map('intval', $ids);

        $result = [];
        foreach ($ids as $id) {
            $aux = $this->getProductSummary($id);

            if (!empty((array) $aux)) {
                $product = new Product($aux['internalId'], false, $this->getContext()->language->id);
                $aux['description'] = $product->description;

                if ($extraProperties == 'tags') {
                    $tags = Tag::getProductTags($aux['internalId']);
                    $aux['tags'] = $tags[$this->getContext()->language->id];
                }
                $attr = $product->hasAttributes();

                if ($attr > 0 || !$product->checkQty(1)) {
                    $aux['addToCartUrl'] = '';
                    $aux['useProductUrl'] = true;
                } else {
                    $auxurl = $this->getContext()->link->getPageLink('cart', true, $this->getContext()->language->id, ['add' => 1, 'id_product' => $aux['internalId'], 'token' => Tools::getToken(false), 'qty' => 1]);
                    $auxurl = Oct8ne::removeHttProtocol($auxurl);
                    $aux['addToCartUrl'] = $auxurl;
                    $aux['useProductUrl'] = false;
                }

                $images = $product->getImages($this->getContext()->language->id);

                $medias = [];

                foreach ($images as $image) {
                    $url_image_type = Configuration::get('OCT_URL_IMG_TYPE');

                    $thumbnail = $this->getContext()->link->getImageLink($product->link_rewrite, $image['id_image']);

                    if ($url_image_type == 2) {
                        if (!isset($base_url)) {
                            $base_url = $this->context->shop->getBaseURL();
                            $base_url = str_replace('http://', '', $base_url);
                            $base_url = str_replace('https://', '', $base_url);
                        }

                        $thumbnail_exploded = explode($base_url, $thumbnail);

                        $thumbnail = $thumbnail_exploded[0] . $base_url . $id . '-' . $thumbnail_exploded[1];
                    }

                    $medias[] = ['url' => $thumbnail];
                }

                $aux['medias'] = $medias;

                $result[] = $aux;
            }
        }
        // $this->logoutContextCustomer();

        return $result;
    }

    /**
     * Obtiene informacion simple sobre el producto. LLama a getSummary
     *
     * @return array
     */
    public function productSummaryOctMeth()
    {
        $this->setContextLanguage();
        $this->setContextCurrency();
        // $this->setContextCustomer();
        $this->setContextCountry();

        $ids = trim(strip_tags(Tools::getValue('productIds')));
        $ids = str_replace(' ', '', $ids);
        $ids = explode(',', $ids);
        $ids = array_map('intval', $ids);

        $result = [];
        foreach ($ids as $id) {
            $aux = $this->getProductSummary($id);

            if (!empty($aux)) {
                $result[] = $aux;
            }
        }
        // $this->logoutContextCustomer();

        return $result;
    }

    /**
     * Devuelve la busqueda de productos por los paremetros de entrada especificados
     *
     * @return array
     */
    public function searchOctMeth()
    {
        $ids_product = [];
        $id_lang = (int) $this->getContext()->language->id;
        $is_filter_applied = false;

        // $this->module->loadLibrary('OctSearchFactory', 'search');

        $this->setContextLanguage();
        $this->setContextCurrency();
        // $this->setContextCustomer();
        $this->setContextCountry();

        $result = [];
        $result['total'] = 0;
        $result['results'] = [];
        $result['filters'] = ['applied' => [], 'available' => []];

        // parametro de busqueda
        $search = pSQL(trim(strip_tags(Tools::getValue('search'))));

        // Page por defecto 1
        $page = (int) pSQL(trim(strip_tags(Tools::getValue('page'))));

        if (empty($page)) {
            $page = 1;
        }

        $pageSize = 10;
        if (!empty($search)) {            // parametro de ordenacion
            $allowedOrderFields = ['position', 'price', 'name'];
            $inputOrderBy = pSQL(Tools::getValue('orderby', 'position'));
            $orderBy = in_array($inputOrderBy, $allowedOrderFields, true) ? $inputOrderBy : 'position';

            $allowedDirections = ['asc', 'desc'];
            $inputDir = pSQL(Tools::getValue('dir', 'asc'));
            $dir = in_array(strtolower($inputDir), $allowedDirections, true) ? strtolower($inputDir) : 'asc';

            // Factoria de motores de busqueda
            $factory = new OctSearchFactory($this->context);
            $searchEngine = Configuration::get('OCT_SEARCH_ENGINE');
            // Motor de busqueda concreto
            $motor = $factory->getInstance($searchEngine);

            if ($motor->supportsPagination()) {
                $pageSize = (int) pSQL(trim(strip_tags(Tools::getValue('pageSize'))));
                if (empty($pageSize)) {
                    // PageSize por defecto 10
                    $pageSize = 10;
                }
                $internalPage = $page;
            } else {
                $pageSize = 999;
                $internalPage = 1;
            }

            // Realizar la busqueda
            $aux = $motor->doSearch($id_lang, $search, $internalPage, $pageSize, $orderBy, $dir);

            if (isset($aux) && $aux['total'] > 0) {
                if ($motor->returnsFullProductData()) {// EcommFinder devuelve todos los datos en la búsqueda
                    $result['total'] = $aux['total'];
                    $result['results'] = $aux['result'];
                } else {
                    $result['total'] = 0;

                    foreach ($aux['result'] as $i => $item) {
                        $product_ok = $this->isProductAvailableByFilter($item['id_product'], $result, $ids_product, $is_filter_applied, $page, $pageSize);

                        if ($product_ok) {
                            array_push($result['results'], $this->getProductSummary((int) $item['id_product']));
                            array_push($ids_product, (int) $item['id_product']);

                            ++$result['total'];
                        }
                    }

                    $offset = ($page - 1) * $pageSize;

                    if ($offset < 0) {
                        $offset = 0;
                    }

                    $result['results'] = array_slice($result['results'], $offset, $pageSize);
                }
            } elseif (isset($aux) && isset($aux['error'])) {
                $result['ErrorMessage'] = $aux['error'];
            }
        } else {
            $this->isProductAvailableByFilter(false, $result, $ids_product, $is_filter_applied, $page, $pageSize);
        }

        // Filtros disponibles
        // --------------------------------------------------------------------------------------------------
        if ($ids_product) {
            $this->getFiltersCategory($result, $ids_product);
            $this->getFiltersManufacturer($result, $ids_product);
            $this->getFiltersAttribute($result, $ids_product);
            $this->getFiltersFeature($result, $ids_product);
            $this->getFiltersStock($result, $ids_product);
        }
        // $this->logoutContextCustomer();

        return $result;
    }

    public function isProductAvailableByFilter($id_product, &$result, &$ids_product, &$is_filter_applied, $page, $pageSize)
    {
        $available = true;
        $filter_available = ['category', 'manufacturer', 'stock'];
        $total_products = 0;
        $queries = [];

        foreach ($_REQUEST as $key => $filter) {
            $key = trim(strip_tags($key));

            if (!is_array($filter)) {
                $filter = explode(',', trim(strip_tags($filter)));
            }

            $filter = array_filter($filter, 'ctype_digit');
            $filter = array_map('intval', $filter);

            // Reemplazar función flecha por función anónima tradicional
            $filter = array_filter($filter, function ($v) { return $v > 0; });

            // Validación adicional defensiva - compatible con PHP < 7.4
            if (empty($filter) || !array_reduce($filter, function ($carry, $v) {
                return $carry && is_int($v);
            }, true)) {
                continue;
            }

            $filterList = implode(',', $filter);
            $count_filter = count($filter);

            $query = new DbQuery();
            $query->select('DISTINCT p.id_product');
            $query->from('product', 'p');
            $query->innerJoin('product_shop', 'ps', 'ps.id_product = p.id_product');

            if ($id_product) {
                $query->where('p.id_product = ' . (int) $id_product);
            }
            $query->where('ps.active = 1');

            if (strpos($key, 'feature_') !== false) {
                $query->innerJoin('feature_product', 'fp', 'fp.id_product = ps.id_product');
                $query->innerJoin('feature_value', 'fv', 'fv.id_feature_value = fp.id_feature_value');
                $query->where('fv.id_feature_value IN (' . $filterList . ')');
            } elseif (strpos($key, 'attribute_') !== false) {
                $query->innerJoin('product_attribute', 'pa', 'pa.id_product = ps.id_product');
                $query->innerJoin('product_attribute_combination', 'pac', 'pac.id_product_attribute = pa.id_product_attribute');
                $query->innerJoin('attribute', 'a', 'a.id_attribute = pac.id_attribute');
                $query->where('a.id_attribute IN (' . $filterList . ')');
            } elseif (in_array($key, $filter_available, true)) {
                if ($key === 'category') {
                    if ($this->include_subcategories) {
                        $categories = $filter;
                        foreach ($filter as $id_category) {
                            $this->getIdsCategoriesRecursive($categories, $id_category);
                        }
                        $filter = array_unique(array_map('intval', $categories));
                        $filterList = implode(',', $filter);
                    }

                    $query->innerJoin('category_product', 'cp', 'cp.id_product = ps.id_product');
                    $query->where('cp.id_category IN (' . $filterList . ')');
                } elseif ($key === 'manufacturer') {
                    $query->where('p.id_manufacturer IN (' . $filterList . ')');
                } elseif ($key === 'stock') {
                    $query->innerJoin('stock_available', 'sa', 'sa.id_product = ps.id_product');
                    $query->where('sa.id_product_attribute = 0');
                    $query->where((int) $filter[0] === 1 ? 'sa.quantity > 0' : 'sa.quantity <= 0');
                }
            }

            if ($count_filter > 1) {
                $query->groupBy('p.id_product');
                $query->having('COUNT(p.id_product) >= ' . (int) $count_filter);
            }

            $queries[] = $query->build();
        }

        if ($queries) {
            $products = [];

            foreach ($queries as $query) {
                $rows = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($query);
                $products[] = array_column($rows, 'id_product');
            }

            if (count($products) > 1) {
                $products = call_user_func_array('array_intersect', $products);
            } else {
                $products = !empty($products[0]) ? $products[0] : [];
            }

            $total_products = count($products);

            if (!$id_product && $products) {
                $ids_product = $products;

                $query = new DbQuery();
                $query->select('p.id_product');
                $query->from('product', 'p');
                $query->innerJoin('product_shop', 'ps', 'ps.id_product = p.id_product');
                $query->where('p.id_product IN (' . implode(',', $products) . ')');
                $query->where('ps.id_shop = ' . (int) $this->context->shop->id);
                $query->where('ps.active = 1');
                $query->limit($pageSize, ($page - 1) * $pageSize);

                $products = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($query);
                $products = array_column($products, 'id_product');
            }

            if ($products && empty($id_product)) {
                foreach ($products as $id_product_tpm) {
                    $result['results'][] = $this->getProductSummary((int) $id_product_tpm);
                    $ids_product[] = (int) $id_product_tpm;
                }

                $result['total'] += $total_products;
            } elseif (!$products) {
                $available = false;
            }

            $is_filter_applied = true;
        }

        return $available;
    }

    private function getLinkRewrite($string)
    {
        if (method_exists('Tools', 'link_rewrite')) {
            $method = 'link_rewrite';

            return Tools::$method($string);
        }

        // Fallback para PS 1.7.8+ y 8.x
        return Tools::str2url($string);
    }

    public function getFiltersFeature(&$result, $ids_product = [])
    {
        $id_lang = $this->getContext()->language->id;

        $filters = [];
        foreach ($_REQUEST as $key => $filter) {
            $key = pSQL(trim(strip_tags($key)));
            $filter = pSQL(trim(strip_tags($filter)));

            if (!empty($filter)) {
                if (strpos($key, 'feature_') !== false) {
                    if (strpos($filter, ',') === false) {
                        array_push($filters, $filter);
                    } else {
                        $filter = explode(',', $filter);

                        foreach ($filter as $value) {
                            array_push($filters, $value);
                        }
                    }
                }
            }
        }

        // Filtros disponibles
        // ----------------------------------------------------
        $query = new DbQuery();
        $query->select('fl.name, f.id_feature, fv.id_feature_value, fvl.value, count(*) as count');
        $query->from('feature', 'f');
        $query->innerJoin('feature_lang', 'fl', 'f.id_feature = fl.id_feature AND fl.id_lang = ' . (int) $id_lang);
        $query->innerJoin('feature_value', 'fv', 'f.id_feature = fv.id_feature');
        $query->innerJoin('feature_value_lang', 'fvl', 'fv.id_feature_value = fvl.id_feature_value AND fl.id_lang = ' . (int) $id_lang);
        $query->innerJoin('feature_product', 'fp', 'fv.id_feature_value = fp.id_feature_value');

        if ($ids_product) {
            $ids_product = array_filter($ids_product, 'is_numeric');
            $ids_product = array_map('intval', $ids_product);
            $query->where('fp.id_product IN (' . implode(',', $ids_product) . ')');
        }
        $query->groupBy('f.id_feature, fv.id_feature_value');

        $result_filter = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($query);

        if (is_array($result_filter) && count($result_filter)) {
            $filters_availables = [];

            foreach ($result_filter as $row) {                // Filtros aplicados
                // ----------------------------------------------------
                if ($filters) {
                    foreach ($filters as $i => $id_filter) {
                        $filter = new FeatureValue((int) $id_filter, $id_lang);

                        if ($filter->id_feature == $row['id_feature'] && $filter->id == $row['id_feature_value']) {
                            if (Validate::isLoadedObject($filter)) {
                                $result['filters']['applied'][] = [
                                    'param' => 'feature_' . $this->getLinkRewrite($row['name']),
                                    'paramLabel' => $row['name'],
                                    'value' => (int) $filter->id,
                                    'valueLabel' => $filter->value,
                                ];

                                unset($filters[$i]);

                                continue 2;
                            }
                        }
                    }
                }
                // ----------------------------------------------------

                $filters_availables[$this->getLinkRewrite($row['name'])]['id_feature_value'] = $row['name'];
                $filters_availables[$this->getLinkRewrite($row['name'])]['name'] = $row['name'];
                $filters_availables[$this->getLinkRewrite($row['name'])]['options'][] = [
                    'value' => (int) $row['id_feature_value'],
                    'valueLabel' => $row['value'],
                    'count' => (int) $row['count'],
                ];
            }

            if ($filters_availables) {
                foreach ($filters_availables as $key => $value) {
                    $filter_available = [];
                    $filter_available['param'] = 'feature_' . $key;
                    $filter_available['paramLabel'] = $value['name'];
                    $filter_available['options'] = $value['options'];

                    array_push($result['filters']['available'], $filter_available);
                }
            }
        }
    }

    public function getFiltersAttribute(&$result, $ids_product = [])
    {
        $id_lang = $this->getContext()->language->id;

        $filters = [];
        foreach ($_REQUEST as $key => $filter) {
            $key = pSQL(trim(strip_tags($key)));
            $filter = pSQL(trim(strip_tags($filter)));

            if (!empty($filter)) {
                if (strpos($key, 'attribute_') !== false) {
                    if (strpos($filter, ',') === false) {
                        array_push($filters, $filter);
                    } else {
                        $filter = explode(',', $filter);

                        foreach ($filter as $value) {
                            array_push($filters, $value);
                        }
                    }
                }
            }
        }

        // Filtros disponibles
        // ----------------------------------------------------
        $query = new DbQuery();
        $query->select('agl.public_name, ag.id_attribute_group, al.id_attribute, al.name, count(*) as count');
        $query->from('attribute_group', 'ag');
        $query->innerJoin('attribute_group_lang', 'agl', 'ag.id_attribute_group = agl.id_attribute_group AND agl.id_lang = ' . (int) $id_lang);
        $query->innerJoin('attribute', 'a', 'a.id_attribute_group = agl.id_attribute_group');
        $query->innerJoin('attribute_lang', 'al', 'a.id_attribute = al.id_attribute AND al.id_lang = ' . (int) $id_lang);
        $query->innerJoin('product_attribute_combination', 'pac', 'pac.id_attribute = a.id_attribute');
        $query->innerJoin('product_attribute', 'pa', 'pa.id_product_attribute = pac.id_product_attribute');

        if ($ids_product) {
            $ids_product = array_filter($ids_product, 'is_numeric');
            $ids_product = array_map('intval', $ids_product);
            $query->where('pa.id_product IN (' . implode(',', $ids_product) . ')');
        }
        $query->groupBy('ag.id_attribute_group, a.id_attribute');

        $result_filter = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($query);

        if (is_array($result_filter) && count($result_filter)) {
            $filters_availables = [];

            foreach ($result_filter as $row) {                // Filtros aplicados
                // ----------------------------------------------------
                if ($filters) {
                    foreach ($filters as $i => $id_filter) {
                        $filterObj = new Attribute((int) $id_filter);

                        if (Validate::isLoadedObject($filterObj)) {
                            // FIX: Usamos json_encode/decode para "borrar" el tipo de objeto.
                            // Esto impide que el validador asuma erróneamente que el array resultante solo tiene la clave 'flags'.
                            // Al hacerlo así, se convierte en un array genérico y permite acceder a cualquier clave.
                            $filterData = json_decode(json_encode($filterObj), true);

                            $filterGroup = isset($filterData['id_attribute_group']) ? $filterData['id_attribute_group'] : null;
                            $filterId = isset($filterData['id']) ? $filterData['id'] : null;
                            $filterName = isset($filterData['name']) ? $filterData['name'] : '';

                            if ($filterGroup == $row['id_attribute_group'] && $filterId == $row['id_attribute']) {
                                $result['filters']['applied'][] = [
                                    'param' => 'attribute_' . $this->getLinkRewrite($row['public_name']),
                                    'paramLabel' => $row['public_name'],
                                    'value' => (int) $filterId,
                                    'valueLabel' => $filterName,
                                ];

                                unset($filters[$i]);
                                continue 2;
                            }
                        }
                    }
                }
                // ----------------------------------------------------

                $filters_availables[$this->getLinkRewrite($row['public_name'])]['id_attribute_group'] = $row['id_attribute_group'];
                $filters_availables[$this->getLinkRewrite($row['public_name'])]['name'] = $row['public_name'];
                $filters_availables[$this->getLinkRewrite($row['public_name'])]['options'][] = [
                    'value' => (int) $row['id_attribute'],
                    'valueLabel' => $row['name'],
                    'count' => (int) $row['count'],
                ];
            }

            if ($filters_availables) {
                foreach ($filters_availables as $key => $value) {
                    $filter_available = [];
                    $filter_available['param'] = 'attribute_' . $key;
                    $filter_available['paramLabel'] = $value['name'];
                    $filter_available['options'] = $value['options'];

                    array_push($result['filters']['available'], $filter_available);
                }
            }
        }
    }

    public function getFiltersCategory(&$result, $ids_product = [])
    {
        $id_lang = $this->getContext()->language->id;
        $filters = Tools::getValue('category', '');
        $filters = explode(',', $filters);

        $filters = array_filter($filters, function ($v) {
            return ctype_digit($v);
        });

        if (empty($filters)) {
            return;
        }
        $filters = array_map('intval', $filters);

        // Filtros disponibles
        // ----------------------------------------------------
        $query = new DbQuery();
        $query->select('cp.id_category, cl.name, count(*) as count');
        $query->from('category_product', 'cp');
        $query->innerJoin('category_lang', 'cl', 'cl.id_category = cp.id_category');
        $query->where('cl.id_lang = ' . (int) $id_lang);

        if ($ids_product) {
            $query->where('cp.id_product IN (' . implode(',', $ids_product) . ')');
        }

        $safeFilters = array_map('intval', (array) $filters);
        $query->where('cp.id_category NOT IN (' . implode(',', $safeFilters) . ',' . (int) Configuration::get('PS_ROOT_CATEGORY') . ')');

        $query->groupBy('cp.id_category');
        $result_filter = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($query);

        if (is_array($result_filter) && count($result_filter)) {
            $filter_available = [];
            $filter_available['param'] = 'category';
            $filter_available['paramLabel'] = $this->module->l('Categories', $this->name_controller);

            foreach ($result_filter as $row) {
                $filter_available['options'][] = [
                    'value' => (int) $row['id_category'],
                    'valueLabel' => $row['name'],
                    'count' => (int) $row['count'],
                ];
            }

            array_push($result['filters']['available'], $filter_available);
        }

        // Filtros aplicados
        // ----------------------------------------------------
        foreach ($filters as $id_filter) {
            $filter = new Category((int) $id_filter, $id_lang);

            if (Validate::isLoadedObject($filter)) {
                $result['filters']['applied'][] = [
                    'param' => 'category',
                    'paramLabel' => $this->module->l('Categories', $this->name_controller),
                    'value' => (int) $filter->id,
                    'valueLabel' => $filter->name,
                ];
            }
            unset($filter);
        }
    }

    public function getFiltersManufacturer(&$result, $ids_product = [])
    {
        $id_lang = $this->getContext()->language->id;
        $filters = trim(strip_tags(Tools::getValue('manufacturer')));
        $filters = explode(',', $filters);

        $filters = array_filter($filters, function ($v) {
            return ctype_digit($v);
        });

        if (empty($filters)) {
            return;
        }
        $filters = array_map('intval', $filters);

        // Filtros disponibles
        // ----------------------------------------------------
        $query = new DbQuery();
        $query->select('m.id_manufacturer, m.name, count(*) as count');
        $query->from('manufacturer', 'm');
        $query->innerJoin('product', 'p', 'p.id_manufacturer = m.id_manufacturer');

        if ($ids_product) {
            $query->where('p.id_product IN (' . implode(',', $ids_product) . ')');
        }

        $query->where('m.id_manufacturer NOT IN (' . implode(',', $filters) . ')');
        $query->groupBy('m.id_manufacturer');
        $result_filter = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($query);

        if (is_array($result_filter) && count($result_filter)) {
            $filter_available = [];
            $filter_available['param'] = 'manufacturer';
            $filter_available['paramLabel'] = $this->module->l('Manufacturers', $this->name_controller);

            foreach ($result_filter as $row) {
                $filter_available['options'][] = [
                    'value' => (int) $row['id_manufacturer'],
                    'valueLabel' => $row['name'],
                    'count' => (int) $row['count'],
                ];
            }

            array_push($result['filters']['available'], $filter_available);
        }

        foreach ($filters as $id_filter) {
            $filter = new Manufacturer((int) $id_filter, $id_lang);

            if (Validate::isLoadedObject($filter)) {
                $result['filters']['applied'][] = [
                    'param' => 'manufacturer',
                    'paramLabel' => $this->module->l('Manufacturers', $this->name_controller),
                    'value' => (int) $filter->id,
                    'valueLabel' => $filter->name,
                ];
            }
            unset($filter);
        }
    }

    public function getFiltersStock(&$result, $ids_product)
    {
        $id_lang = $this->getContext()->language->id;
        $filters = pSQL(trim(strip_tags(Tools::getValue('stock'))));

        // Filtros disponibles
        // ----------------------------------------------------
        $query = new DbQuery();
        $query->select('count(*) as count');
        $query->from('stock_available', 'sa');
        $query->where('sa.quantity > 0');
        $query->where('sa.id_product_attribute = 0');

        if ($ids_product) {
            $ids_product = array_filter($ids_product, 'is_numeric');
            $ids_product = array_map('intval', $ids_product);
            $query->where('sa.id_product IN (' . implode(',', $ids_product) . ')');
        }
        $with_stock = Db::getInstance(_PS_USE_SQL_SLAVE_)->getValue($query);

        $query = new DbQuery();
        $query->select('count(*) as count');
        $query->from('stock_available', 'sa');
        $query->where('sa.quantity <= 0');
        $query->where('sa.id_product_attribute = 0');

        if ($ids_product) {
            $query->where('sa.id_product IN (' . implode(',', $ids_product) . ')');
        }
        $without_stock = Db::getInstance(_PS_USE_SQL_SLAVE_)->getValue($query);

        $filter_available = [];
        $filter_available['param'] = 'stock';
        $filter_available['paramLabel'] = $this->module->l('Stock', $this->name_controller);

        if ((int) $filters === 1) {
            $filter_available['options'][] = [
                'value' => -1,
                'valueLabel' => $this->module->l('No', $this->name_controller),
                'count' => (int) $without_stock,
            ];
        } elseif (Tools::isSubmit('stock')) {
            $filter_available['options'][] = [
                'value' => 1,
                'valueLabel' => $this->module->l('Yes', $this->name_controller),
                'count' => (int) $with_stock,
            ];
        } else {
            $filter_available['options'][] = [
                'value' => -1,
                'valueLabel' => $this->module->l('No', $this->name_controller),
                'count' => (int) $without_stock,
            ];
            $filter_available['options'][] = [
                'value' => 1,
                'valueLabel' => $this->module->l('Yes', $this->name_controller),
                'count' => (int) $with_stock,
            ];
        }

        array_push($result['filters']['available'], $filter_available);

        // Filtros aplicados
        // ----------------------------------------------------
        if ((int) $filters === 1) {
            $result['filters']['applied'][] = [
                'param' => 'stock',
                'paramLabel' => $this->module->l('Stock', $this->name_controller),
                'value' => 1,
                'valueLabel' => $this->module->l('Yes', $this->name_controller),
            ];
        } elseif (Tools::isSubmit('stock')) {
            $result['filters']['applied'][] = [
                'param' => 'stock',
                'paramLabel' => $this->module->l('Stock', $this->name_controller),
                'value' => -1,
                'valueLabel' => $this->module->l('No', $this->name_controller),
            ];
        }
    }

    /**
     * Obtiene informacion de productos relacionados con el indicado
     *
     * @return array
     */
    public function productRelatedOctMeth()
    {
        $this->setContextLanguage();
        $this->setContextCurrency();
        // $this->setContextCustomer();
        $this->setContextCountry();

        $result = [];
        $result['total'] = 0;
        $result['results'] = [];

        $productId = pSQL(trim(strip_tags(Tools::getValue('productId'))));

        if (!empty($productId)) {
            $product = new Product((int) $productId, false, $this->getContext()->language->id);

            // si no se ha cargado devuelvo vacio
            if (!Validate::isLoadedObject($product)) {
                return $result;
            }

            $category = $product->id_category_default;

            // cargo el objeto categorias
            $category = new Category($category, $this->getContext()->language->id);

            // si esta cargado
            if (Validate::isLoadedObject($category)) {
                // compruebo el total,
                $total = $category->getProducts($this->getContext()->language->id, 1, 100, 'position', 'asc', true, true);

                if ((int) $total > 1) {
                    $result['total'] = $total - 1;
                    $products = $category->getProducts($this->getContext()->language->id, 1, 100, 'position', 'asc', false, true);
                    $products = array_filter($products, function ($v) use ($productId) {                        // $this->logoutContextCustomer();
                        return $v['id_product'] != $productId;
                    });

                    foreach ($products as $item) {
                        array_push($result['results'], $this->getProductSummary($item['id_product']));
                    }
                    // $this->logoutContextCustomer();

                    return $result;
                } else {                    // $this->logoutContextCustomer();
                    return $result;
                }
            } else {                // $this->logoutContextCustomer();
                return $result;
            }
        } else {            // $this->logoutContextCustomer();
            return $result;
        }
    }

    /**
     * @return array
     *               Obtiene informacion sobre el usuario
     */
    public function customerDataOctMeth()
    {
        $this->setContextLanguage();
        $this->setContextCurrency();
        // $this->setContextCustomer();
        $this->setContextCountry();

        $result = [];
        $result['id'] = '';
        $result['firstName'] = '';
        $result['lastName'] = '';
        $result['email'] = '';
        $result['wishlist'] = [];
        $result['cart'] = [];

        $customer = $this->getContext()->customer;

        if (!empty($customer) && Validate::isLoadedObject($customer) && $customer->isLogged() && !$customer->deleted) {
            $result['id'] = $customer->id;
            $result['firstName'] = $customer->firstname;
            $result['lastName'] = $customer->lastname;
            $result['email'] = $customer->email;

            // si esta el modulo de lista blanca activo
            if (Module::isEnabled('blockwishlist') && class_exists('WishList')) {
                Module::getInstanceByName('blockwishlist');

                // obtiene toda la listas de deseis que tiene el usuario
                $wishlist = WishList::getByIdCustomer($customer->id);

                // si tiene una lista de deseos en cookies
                if (isset($this->context->cookie->id_wishlist) && $this->context->cookie->id_wishlist !== '' && !empty($wishlist)) {
                    // cojo solo la de la cookie que sera la ultima
                    $wishlist = array_filter($wishlist, function ($v) {
                        return $v['id_wishlist'] == $this->getContext()->cookie->id_wishlist;
                    });

                    if (!empty($wishlist)) {
                        // obtengo el primer elemento
                        $wishlist = array_shift($wishlist);

                        // recojo los productos
                        $products = WishList::getProductByIdCustomer((int) $wishlist['id_wishlist'], $customer->id, $this->context->language->id, null, true);

                        if (!empty($products)) {
                            foreach ($products as $product) {
                                // los meto en el resultado
                                array_push($result['wishlist'], $this->getProductSummary($product['id_product']));
                            }
                        }
                    }
                }
            }
        }

        $cart = $this->getContext()->cart;

        if (!empty($cart)) {
            $in_cart = $cart->getProducts();

            foreach ($in_cart as $in) {
                $aux = $this->getProductSummary($in['id_product']);
                $aux['qty'] = $in['cart_quantity'];

                array_push($result['cart'], $aux);
            }
        }
        // $this->logoutContextCustomer();

        return $result;
    }

    /**
     * Agregar productos a la lista blanca
     *
     * @return bool
     *
     * @throws PrestaShopException
     */
    public function addToWishListOctMeth()
    {
        $result = false;

        try {
            if (Module::isEnabled('blockwishlist') && class_exists('WishList')) {
                $customer = $this->getContext()->customer;

                if (!empty($customer) && Validate::isLoadedObject($customer) && $customer->isLogged() && !$customer->deleted) {// Comprobamos que el usuario es accesible
                    $ids = trim(strip_tags(Tools::getValue('productIds')));
                    $ids = explode(',', $ids);
                    $ids = array_map('intval', $ids);

                    Module::getInstanceByName('blockwishlist');

                    $context = $this->getContext();

                    if (!isset($context->cookie->id_wishlist) || $context->cookie->id_wishlist == '') {
                        $wishlist = new WishList();
                        $wishlist->id_shop = $context->shop->id;
                        $wishlist->id_shop_group = $context->shop->id_shop_group;
                        $wishlist->default = 1;

                        $mod_wishlist = new BlockWishList();
                        // FIX: Comprobamos si la propiedad existe o usamos un valor por defecto
                        $wishlist->name = isset($mod_wishlist->default_wishlist_name) ? $mod_wishlist->default_wishlist_name : 'Default';
                        $wishlist->id_customer = (int) $context->customer->id;
                        list($us, $s) = explode(' ', microtime());
                        srand((int) $s * (int) $us);
                        $wishlist->token = Tools::strtoupper(Tools::substr(sha1(uniqid((string) rand(), true) . _COOKIE_KEY_ . $context->customer->id), 0, 16));
                        $wishlist->add();
                        $context->cookie->id_wishlist = (int) $wishlist->id;
                    }

                    foreach ($ids as $id_product) {
                        $id_product_attribute = (int) Product::getDefaultAttribute($id_product);
                        WishList::addProduct($context->cookie->id_wishlist, $context->customer->id, $id_product, $id_product_attribute, 1);
                    }

                    $result = true;
                }
            } else {
                $result = false;
            }
        } catch (Exception $e) {
            $result = false;
            // $this->module->logException($e);
        }

        return $result;
    }

    /**
     * Obtiene el carro de compra
     *
     * @return array
     */
    public function getCartOctMeth()
    {
        $result = [];
        $result['price'] = '0';
        $result['finalPrice'] = '0';
        $result['currency'] = '';
        $result['totalItems'] = '0';
        $result['cart'] = [];

        return $result;
    }

    /**
     * Obtiene un informe de los productos vendidos a traves de oct8ne entre dos fechas
     *
     * @return array
     */
    public function getSalesReportOctMeth()
    {
        $result = [];

        return $result;
    }

    /**
     * Devuelve informacion sobre el modulo
     *
     * @return array
     */
    public function getAdapterInfoOctMeth()
    {
        $result = [];
        $result['platform'] = 'Prestashop';
        $result['adapterName'] = 'Oct8ne official adapter for Prestashop';
        $result['adapterVersion'] = $this->module->version;
        $result['developedBy'] = 'Oct8ne Inc';
        $result['supportUrl'] = '';
        $result['apiVersion'] = '2.5';
        $result['enabled'] = $this->module->active == 1;

        $factory = new OctSearchFactory($this->context);

        $searchEngine = Configuration::get('OCT_SEARCH_ENGINE');
        $motor = $factory->getInstance($searchEngine);

        $engineName = $motor->getEngineInfo();

        $result['searchEngine'] = $engineName;

        return $result;
    }

    /**
     * Obtiene todos los pedidos de un cliente por email.
     * Pedidos validos o no, son devueltos.
     *
     * @return array
     */
    public function getOrdersOctMeth()
    {
        $filterByField = pSQL(Configuration::get('OTC_ORDER_DETAILS_BY'));

        if (!$filterByField) {
            $filterByField = 'reference';
        }

        $customerEmail = pSQL(Tools::getValue('customerEmail'));
        $apiToken = pSQL(Tools::getValue('apiToken'));

        // Page por defecto 1
        $page = (int) pSQL(trim(strip_tags(Tools::getValue('page'))));

        if (empty($page)) {
            $page = 1;
        }

        // PageSize por defecto 10
        $pageSize = (int) pSQL(trim(strip_tags(Tools::getValue('pageSize'))));

        if (empty($pageSize)) {
            $pageSize = 10;
        }

        $offset = $pageSize * ($page - 1);

        $this->setContextLanguage();
        $this->setContextCurrency();
        // $this->setContextCustomer();
        $this->setContextCountry();

        $in_token = Configuration::get('OCT_API_TOKEN');

        $result = [];

        if ($in_token == $apiToken && !empty($customerEmail)) {
            $id_customer = Customer::customerExists($customerEmail, true);

            if ($id_customer) {
                $sqlOrders = new DbQuery();
                $sqlOrders->select('o.*, osl.`name` AS order_state');
                $sqlOrders->from('orders', 'o');
                $sqlOrders->leftJoin('order_state_lang', 'osl', 'osl.`id_order_state` = o.`current_state` AND osl.`id_lang` = ' . (int) $this->getContext()->language->id);
                $sqlOrders->where('o.`id_customer` = ' . (int) $id_customer);
                $sqlOrders->where('o.id_shop = ' . (int) $this->context->shop->id);
                $sqlOrders->orderBy('o.`id_order` DESC');
                $sqlOrders->orderBy('o.`date_add` DESC');
                $sqlOrders->limit($pageSize, $offset);
                $orders = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($sqlOrders);

                if ($orders) {
                    foreach ($orders as $order) {
                        $currency = Currency::getCurrency($order['id_currency']);

                        $result[] = [
                            'date' => $order['date_add'],
                            'reference' => $order[$filterByField],
                            'total' => $this->formatPrice($order['total_paid']),
                            'currency' => $currency['iso_code'],
                            'labelState' => $order['order_state'],
                            'deliveryDate' => '',
                        ];
                    }
                }
            }
            // $this->logoutContextCustomer();
        }

        return $result;
    }

    /**
     * Obtiene todos los pedidos de un cliente por email.
     * Pedidos validos o no, son devueltos.
     *
     * @return array
     */
    public function getOrderDetailsOctMeth()
    {
        $filterByField = pSQL(Configuration::get('OTC_ORDER_DETAILS_BY'));

        if (!$filterByField) {
            $filterByField = 'reference';
        }

        $reference = pSQL(Tools::getValue('reference'));
        $apiToken = pSQL(Tools::getValue('apiToken'));

        $this->setContextLanguage();
        $this->setContextCurrency();
        // $this->setContextCustomer();
        $this->setContextCountry();

        $in_token = pSQL(Configuration::get('OCT_API_TOKEN'));

        $result = [];

        if ($in_token == $apiToken && !empty($reference)) {
            $sqlOrder = new DbQuery();
            $sqlOrder->select('o.*, osl.name AS order_state, IFNULL(c.name, "Carrier not exist") as carrier_name, oc.tracking_number, c.url as tracking_url');
            $sqlOrder->from('orders', 'o');
            $sqlOrder->innerJoin('order_carrier', 'oc', 'oc.id_order = o.id_order');
            $sqlOrder->leftJoin('carrier', 'c', 'c.id_carrier = oc.id_carrier');
            $sqlOrder->leftJoin('order_state_lang', 'osl', 'osl.id_order_state = o.current_state AND osl.id_lang = ' . (int) $this->getContext()->language->id);

            if (Tools::substr($filterByField, 0, 3) == 'id_' || $filterByField == 'id') {
                $sqlOrder->where('o.' . bqSQL($filterByField) . '= ' . (int) pSQL($reference));
            } else {
                $sqlOrder->where('o.' . bqSQL($filterByField) . '= "' . pSQL($reference) . '"');
            }

            $sqlOrder->where('o.id_shop = ' . (int) $this->context->shop->id);
            $sqlOrder->orderBy('o.id_order');
            $sqlOrder->orderBy('o.date_add DESC');
            $order = Db::getInstance(_PS_USE_SQL_SLAVE_)->getRow($sqlOrder);

            if ($order) {
                $currency = Currency::getCurrency($order['id_currency']);
                $products = [];

                $sqlProductsOrder = new DbQuery();
                $sqlProductsOrder->from('order_detail', 'o');
                $sqlProductsOrder->where('id_order = ' . (int) $order['id_order']);
                $productsOrder = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($sqlProductsOrder);

                if ($productsOrder) {
                    foreach ($productsOrder as $product) {
                        $products[] = [
                            'quantity' => $product['product_quantity'],
                            'name' => Product::getProductName(
                                $product['product_id'],
                                $product['product_attribute_id'],
                                $this->getContext()->language->id
                            ),
                        ];
                    }
                }

                $trackingNumber = $order['tracking_number'];
                $trackingURL = '';

                if (!empty($trackingNumber)) {
                    $trackingURL = Tools::str_replace_once('@', $trackingNumber, $order['tracking_url']);
                }

                $result = [
                    'date' => $order['date_add'],
                    'reference' => $order[$filterByField],
                    'total' => $this->formatPrice($order['total_paid']),
                    'currency' => $currency['iso_code'],
                    'labelState' => $order['order_state'],
                    'deliveryDate' => '',
                    'carrier' => $order['carrier_name'],
                    'trackingNumber' => $trackingNumber,
                    'trackingURL' => $trackingURL,
                    'products' => $products,
                ];

                $messages = CustomerMessage::getMessagesByOrderId((int) $order['id_order']);

                if ($messages) {
                    foreach ($messages as $i => $message) {
                        $result['comments'][$i]['message'] = $message['message'];
                        $result['comments'][$i]['customer'] = empty($message['id_employee']);
                    }
                }
            }
            // $this->logoutContextCustomer();
        }

        return $result;
    }

    /**
     * Obtiene todas las categorias activas del catalogo
     *
     * @return array
     */
    public function getCategoriesOctMeth()
    {
        $this->setContextLanguage();
        $this->setContextCurrency();
        // $this->setContextCustomer();
        $this->setContextCountry();

        $categories = [];

        $this->getCategoriesRecursive($categories, Configuration::get('PS_HOME_CATEGORY'));
        // $this->logoutContextCustomer();

        return $categories;
    }

    private function getCategoriesRecursive(&$categories, $id_category)
    {
        $sql = new DbQuery();
        $sql->from('category', 'c');
        $sql->innerJoin('category_lang', 'cl', 'c.id_category = cl.id_category AND cl.id_lang = ' . (int) $this->getContext()->language->id . ' AND cl.id_shop = ' . (int) $this->context->shop->id);
        $sql->innerJoin('category_shop', 'cs', 'c.id_category = cs.id_category AND cs.id_shop = ' . (int) $this->context->shop->id);
        $sql->where('c.id_parent = ' . (int) $id_category);
        $result = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($sql);

        if ($result) {
            foreach ($result as $value) {
                $children = [];
                $this->getCategoriesRecursive($children, $value['id_category']);

                if ($this->include_subcategories && is_array($children) && !empty($children)) {
                    $countProducts = 0;
                    foreach ($children as $child) {
                        $countProducts += $child['productCount'];
                    }
                } else {
                    $sql1 = new DbQuery();
                    $sql1->select('COUNT(id_category)');
                    $sql1->from('category_product', 'c');
                    $sql1->where('c.id_category = ' . (int) $value['id_category']);
                    $countProducts = (int) Db::getInstance(_PS_USE_SQL_SLAVE_)->getValue($sql1);
                }

                $categories[] = [
                    'id' => (int) $value['id_category'],
                    'parentId' => (int) $value['id_parent'],
                    'name' => $value['name'],
                    'productCount' => $countProducts,
                    'children' => $children,
                ];
            }
        }
    }

    private function getIdsCategoriesRecursive(&$categories, $id_category)
    {
        $sql = new DbQuery();
        $sql->from('category', 'c');
        $sql->innerJoin('category_shop', 'cs', 'c.id_category = cs.id_category AND cs.id_shop = ' . (int) $this->context->shop->id);
        $sql->where('c.id_parent = ' . (int) $id_category);

        $result = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($sql);

        if ($result) {
            foreach ($result as $value) {
                $children = [];

                array_push($categories, (int) $value['id_category']);

                $this->getIdsCategoriesRecursive($categories, (int) $value['id_category']);
            }
        }
    }

    /**
     * Obtiene los productos de la categoria enviada.
     *
     * @return array
     */
    public function getProductsByCategoryOctMeth()
    {
        $id_category = pSQL(Tools::getValue('idCategory'));

        $this->setContextLanguage();
        $this->setContextCurrency();
        // $this->setContextCustomer();
        $this->setContextCountry();

        $products = [];

        $sql = new DbQuery();
        $sql->select('id_product');
        $sql->from('category_product', 'cp');
        $sql->where('cp.id_category = ' . (int) $id_category);
        $result = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($sql);

        if ($result) {
            foreach ($result as $value) {
                $aux = $this->getProductSummary((int) $value['id_product']);

                if (!empty($aux)) {
                    $products[] = $aux;
                }
            }
        }
        // $this->logoutContextCustomer();

        return [
            'total' => count($products),
            'result' => $products,
        ];
    }
}
