<?php
/**
 * Prestashop module for Oct8ne
 *
 * @author    oct8ne.com
 * @copyright 2025 Oct8ne
 * @license   Commercial license see license.txt *
 *
 * @category  Prestashop
 * @category  Module
 */
if (!defined('_PS_VERSION_')) {
    exit;
}

require_once dirname(__FILE__) . '/AbstractSearchEngine.php';

class OctEmpathySearch extends AbstractSearchEngine
{
    private $context;

    public function __construct(Context $context)
    {
        $this->context = $context;
    }

    private function getContext()
    {
        return $this->context;
    }

    public function supportsPagination()
    {
        return true;
    }

    public function getEngineInfo()
    {
        $info = [
            'engine' => 'Empathy',
            'languages' => [],
        ];

        $id_shop = (int) $this->context->shop->id;
        $languages = Language::getLanguages(true, $id_shop);

        foreach ($languages as $lang) {
            $iso = $lang['language_code'];

            // Get config values
            $instance = Configuration::get(Oct8ne::$EMPATHY_INSTANCE . '_' . $iso);
            $site = Configuration::get(Oct8ne::$EMPATHY_SITE_ID . '_' . $iso);

            if (empty($instance)) {
                $instance = Configuration::get(Oct8ne::$EMPATHY_INSTANCE);
            }
            if (empty($site)) {
                $site = Configuration::get(Oct8ne::$EMPATHY_SITE_ID);
            }

            $info['languages'][$iso] = [
                'instance' => $this->maskValue($instance),
                'site' => $this->maskValue($site),
            ];
        }

        return $info;
    }

    private function maskValue($value)
    {
        if (!$value) {
            return '';
        }
        if (strlen($value) <= 6) {
            return str_repeat('X', strlen($value));
        }

        return substr($value, 0, 4) . str_repeat('X', strlen($value) - 4);
    }

    public function doSearch($language, $search, $page, $pageSize, $orderBy, $dir)
    {
        // --- FIX API LIMIT (Empathy max rows is 500) ---
        if ($pageSize > 500) {
            $pageSize = 500;
        }

        $context = $this->getContext();
        $langObj = new Language((int) $language);
        $langCode = $langObj->language_code;

        // 1. GET CONFIGURATION
        $customEndpoint = Configuration::get(Oct8ne::$EMPATHY_API_ENDPOINT . '_' . $langCode);
        if (empty($customEndpoint)) {
            $customEndpoint = Configuration::get(Oct8ne::$EMPATHY_API_ENDPOINT);
        }

        $instance = Configuration::get(Oct8ne::$EMPATHY_INSTANCE . '_' . $langCode);
        if (empty($instance)) {
            $instance = Configuration::get(Oct8ne::$EMPATHY_INSTANCE);
        }

        $site = Configuration::get(Oct8ne::$EMPATHY_SITE_ID . '_' . $langCode);
        if (empty($site)) {
            $site = Configuration::get(Oct8ne::$EMPATHY_SITE_ID);
        }

        $store = Configuration::get(Oct8ne::$EMPATHY_STORE . '_' . $langCode);
        if (empty($store)) {
            $store = Configuration::get(Oct8ne::$EMPATHY_STORE);
        }

        $lang = Configuration::get(Oct8ne::$EMPATHY_LANG . '_' . $langCode);
        if (empty($lang)) {
            $lang = Configuration::get(Oct8ne::$EMPATHY_LANG);
        }

        // 2. PAGINATION (Convert page to start row)
        $start = ((int) $page - 1) * (int) $pageSize;

        // 3. BUILD URL
        $queryText = urlencode($search);
        $baseUrl = '';

        if (!empty($customEndpoint)) {
            $baseUrl = rtrim($customEndpoint, '/') . '/' . $instance . '/search';
        } else {
            if (empty($instance)) {
                return ['total' => 0, 'result' => []];
            }
            $baseUrlFixed = 'https://api.empathy.co/search/v1/query/';
            $baseUrl = $baseUrlFixed . $instance . '/search';
        }

        // AÑADIR PARÁMETROS OBLIGATORIOS (Query, Start, Lang, Store)
        $url = $baseUrl . "?query={$queryText}&start={$start}&rows={$pageSize}";

        if (!empty($store) && $store !== 'default') {
            $url .= '&store=' . urlencode($store);
        }
        $url .= '&lang=' . $lang;

        // 4. HTTP REQUEST
        $curl = curl_init($url);
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($curl, CURLOPT_TIMEOUT, 4);
        curl_setopt($curl, CURLOPT_IPRESOLVE, CURL_IPRESOLVE_V4);
        curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($curl, CURLOPT_SSL_VERIFYHOST, true);

        $response = curl_exec($curl);

        if ($response === false) {
            curl_close($curl);

            return ['total' => 0, 'result' => []];
        }

        curl_close($curl);

        $data = json_decode($response);

        // 5. PARSE RESPONSE
        if (!$data || empty($data->catalog)) {
            return ['total' => 0, 'result' => []];
        }

        $total = 0;
        if (isset($data->catalog->numFound)) {
            $total = (int) $data->catalog->numFound;
        } elseif (isset($data->catalog->pagination->total)) {
            $total = (int) $data->catalog->pagination->total;
        } elseif (isset($data->catalog->total)) {
            $total = (int) $data->catalog->total;
        }

        // --- LOGIC FIX: Handle 'content' vs 'products' ---
        $items = [];
        if (isset($data->catalog->content)) {
            $items = $data->catalog->content;
        } elseif (isset($data->catalog->products)) {
            $items = $data->catalog->products;
        }

        $products = [];

        if (!empty($items)) {
            foreach ($items as $p) {
                if (isset($p->id)) {
                    $products[] = [
                        'id_product' => (int) $p->id,
                    ];
                }
            }
        }

        return [
            'total' => $total,
            'result' => $products,
        ];
    }
}
