<?php
/**
 * Kimera Search Engine for Oct8ne
 * Integration with Kimera Technologies API v3.0
 *
 * @author    Oct8ne
 * @copyright 2025 Oct8ne
 * @license   Commercial license see license.txt
 */
if (!defined('_PS_VERSION_')) {
    exit;
}

require_once dirname(__FILE__) . '/AbstractSearchEngine.php';

class OctKimeraSearch extends AbstractSearchEngine
{
    private $context;

    public function __construct(Context $context)
    {
        $this->context = $context;
    }

    private function getContext()
    {
        return $this->context;
    }

    public function supportsPagination()
    {
        return true;
    }

    /**
     * Returns engine configuration info for the module settings page
     */
    public function getEngineInfo()
    {
        $info = [
            'engine' => 'Kimera',
            'languages' => [],
        ];

        $id_shop = (int) $this->context->shop->id;
        $languages = Language::getLanguages(true, $id_shop);

        foreach ($languages as $lang) {
            // CRITICAL: Use 'language_code' (e.g., 'en-us') to match the database keys
            // The ISO code ('en') is not enough because your DB uses full locale
            $code = $lang['language_code'];

            // 1. Retrieve configuration using the full locale suffix
            $store_id = Configuration::get(Oct8ne::$KIMERA_STORE_ID . '_' . $code);
            $api_key = Configuration::get(Oct8ne::$KIMERA_API_KEY . '_' . $code);

            // 2. Fallback strategies if specific config is missing
            if (empty($store_id)) {
                // Try global fallback
                $store_id = Configuration::get(Oct8ne::$KIMERA_STORE_ID);
            }
            if (empty($api_key)) {
                $api_key = Configuration::get(Oct8ne::$KIMERA_API_KEY);
            }

            $info['languages'][$code] = [
                'instance' => $this->maskValue($store_id),
                'site' => $this->maskValue($api_key),
            ];
        }

        return $info;
    }

    /**
     * Executes the search against Kimera API
     * Fixed to support Bearer Auth and correct Locale handling
     */
    public function doSearch($language, $search, $page, $pageSize, $orderBy, $dir)
    {
        // 1. Context & Config
        $langObj = new Language((int) $language);
        $suffix = '_' . $langObj->language_code;

        $apiKey = Configuration::get(Oct8ne::$KIMERA_API_KEY . $suffix) ?: Configuration::get(Oct8ne::$KIMERA_API_KEY);
        $kimeraLang = Configuration::get(Oct8ne::$KIMERA_LANG . $suffix) ?: $langObj->iso_code;

        // --> RETRIEVE INPUT MODE <--
        $inputMode = Configuration::get(Oct8ne::$KIMERA_INPUT_MODE . $suffix);

        // Strict check because "0" is valid but empty() considers it false
        if ($inputMode === false || $inputMode === null || $inputMode === '') {
            $inputMode = '0.5'; // Default to Hybrid
        }

        $userEndpoint = Configuration::get(Oct8ne::$KIMERA_API_ENDPOINT . $suffix);
        $baseUrl = $userEndpoint ?: 'https://api.kimeratechnologies.com/API/v3_0/DB/query';

        // 2. Validation
        if (empty($apiKey)) {
            return ['total' => 0, 'result' => []];
        }
        // Force HTTPS
        $baseUrl = str_replace('http://', 'https://', rtrim($baseUrl, '/'));

        // 3. API v3 Parameters
        $queryParams = [
            'image_query' => $search,
            'elements_per_page' => (int) $pageSize,
            'page_number' => (int) $page,
            'language' => $kimeraLang,
            'element_search_mode' => 'True',
            'input_mode' => $inputMode, // <--- USE VARIABLE
        ];

        $url = $baseUrl . '?' . http_build_query($queryParams);

        // 4. cURL (Bearer Token)
        $curl = curl_init($url);
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($curl, CURLOPT_TIMEOUT, 5);

        $headers = [
            'Content-Type: application/json',
            'Authorization: Bearer ' . $apiKey,
        ];

        curl_setopt($curl, CURLOPT_HTTPHEADER, $headers);

        $response = curl_exec($curl);
        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);

        // 5. Error Handling
        if ($httpCode !== 200 || !$response) {
            return ['total' => 0, 'result' => []];
        }

        $data = json_decode($response);

        // 6. Parse Response
        if (!$data || !isset($data->message) || !is_array($data->message)) {
            return ['total' => 0, 'result' => []];
        }

        $products = [];
        foreach ($data->message as $p) {
            if (isset($p->id)) {
                $products[] = ['id_product' => (int) $p->id];
            }
        }

        // 7. Pagination Logic
        $countReceived = count($products);

        if ($countReceived >= $pageSize) {
            $total = ($page * $pageSize) + 1;
        } else {
            $total = (($page - 1) * $pageSize) + $countReceived;
        }

        return ['total' => $total, 'result' => $products];
    }
}
